<?php

namespace App\Http\Controllers;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Illuminate\Support\Facades\Validator;
use App\Models\Admission;
use App\Models\ArmyRecord;
use App\Models\Subject;
use App\Models\School;
use App\Imports\AdmissionsImport;
use App\Models\Course;
use App\Models\Intake;
use App\Models\Bn;
use App\Models\Phase;
use App\Models\AdmissionPhase;
use App\Models\Coy;
use App\Models\Pl;
use App\Models\Sec;
use App\Models\Student;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;
use App\Exports\AdmissionTemplateExport;
use App\Exports\AdmissionsExport;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Auth;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Validation\Rule;
use Intervention\Image\Facades\Image;
class AdmissionController extends Controller
{

    public function export(Request $request)
    {
        $fileName = 'admissions_' . date('Y-m-d_H-i-s') . '.xlsx';
        return Excel::download(new AdmissionsExport($request->all()), $fileName);
    }

    public function printPDF(Request $request)
    {
        $query = Admission::with(['school', 'course']);

        // Apply filters
        if ($request->school_id) {
            $query->where('school_id', $request->school_id);
        }
        if ($request->course_id) {
            $query->where('course_id', $request->course_id);
        }
        if ($request->bn) {
            $query->where('bn', $request->bn);
        }
        if ($request->coy) {
            $query->where('coy', $request->coy);
        }
        if ($request->pl) {
            $query->where('pl', $request->pl);
        }
        if ($request->sec) {
            $query->where('sec', $request->sec);
        }
        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('army_number', 'like', '%'.$request->search.'%')
                  ->orWhere('name', 'like', '%'.$request->search.'%');
            });
        }

        $admissions = $query->get();

        // Load the view and generate the PDF
        $pdf = PDF::loadView('admissions.print', [
            'admissions' => $admissions,
            'filters' => $request->all()
        ]);

        // Manually set public path for DomPDF if necessary
        $pdf->setOptions([
            'public_path' => realpath(base_path('army_project/public')) // Ensure DomPDF uses the correct public path
        ]);

        // Check if preview is requested
        if ($request->has('preview')) {


            return $pdf->stream('admissions_preview.pdf');
        }

        return $pdf->download('admissions_'.date('Y-m-d_H-i-s').'.pdf');
    }








public function bulkDelete(Request $request)
{
    // Get selected admission IDs
    $ids = $request->input('selected_ids');

    // Check if any ID is selected
    if (!$ids) {
        return redirect()->route('admissions.index')->with('error', 'No records selected.');
    }

    // Delete selected admissions
    Admission::whereIn('id', $ids)->delete();

    return redirect()->route('admissions.index')->with('success', 'Selected admissions deleted successfully.');
}

    public function importForm()
    {
        return view('admissions.import'); // Blade view for the form
    }

public function import(Request $request)
{
    try {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,csv'
        ]);

        if (!$request->hasFile('file') || !$request->file('file')->isValid()) {
            return redirect()->route('admissions.index')->with('error', 'Invalid file upload.');
        }

        $import = new AdmissionsImport();
        Excel::import($import, $request->file('file'));

        $stats = $import->getImportStats();
        
        $message = sprintf(
            'Import completed: %d new records imported, %d records updated, %d records skipped.',
            $stats['imported'],
            $stats['updated'],
            $stats['skipped']
        );

        return redirect()->route('admissions.index')->with('success', $message);

    } catch (\Maatwebsite\Excel\Validators\ValidationException $e) {
        $failures = $e->failures();
        $errorMessages = [];
        
        foreach ($failures as $failure) {
            $errorMessages[] = "Row {$failure->row()}: " . implode(', ', $failure->errors());
        }
        
        return redirect()->route('admissions.index')
            ->with('error', 'Validation errors occurred:')
            ->with('errorDetails', $errorMessages);
            
    } catch (\Exception $e) {
        \Log::error('Import error: ' . $e->getMessage());
        \Log::error('File: ' . $e->getFile() . ' Line: ' . $e->getLine());
        
        return redirect()->route('admissions.index')
            ->with('error', 'Import failed: ' . $e->getMessage());
    }
}

    public function downloadTemplate()
    {
        try {
            // Use the export class to download the template
            return Excel::download(new AdmissionTemplateExport, 'admission_template.xlsx');
        } catch (\Exception $e) {
            // Log the error if anything goes wrong
            Log::error('Error generating Excel template: ' . $e->getMessage());
            return response()->json(['error' => 'An error occurred while generating the file.'], 500);
        }
    }


public function plList(Request $request, $type)
{
    // Initialize query with eager loading to reduce N+1 query problems
    // The global scope automatically excludes decommissioned admissions (is_decommissioned != 1)
    $query = Admission::with(['school', 'course', 'student']);
    
    // Search filter
    if ($request->filled('search')) {
        $query->where(function($q) use ($request) {
            $search = '%' . $request->input('search') . '%';
            $q->where('army_number', 'LIKE', $search)
              ->orWhere('rank', 'LIKE', $search)
              ->orWhere('name', 'LIKE', $search);
        });
    }
    
    // Apply filters with validation to prevent SQL injection
    if ($request->filled('school_id') && is_numeric($request->school_id)) {
        $query->where('school_id', $request->school_id);
    }
    if ($request->filled('course_id') && is_numeric($request->course_id)) {
        $query->where('course_id', $request->course_id);
    }
    
    // Military hierarchy filters
    if ($request->filled('bn')) {
        $query->where('bn', $request->bn);
    }
    if ($request->filled('coy')) {
        $query->where('coy', $request->coy);
        
        // If company is selected but battalion isn't, get the related battalion
        if (!$request->filled('bn')) {
            $coyRecord = Coy::where('name', $request->coy)->first();
            if ($coyRecord && $coyRecord->bn) {
                $query->where('bn', $coyRecord->bn->name);
            }
        }
    }
    if ($request->filled('pl')) {
        $query->where('pl', $request->pl);
        
        // If platoon is selected but company/battalion aren't, get the related company/battalion
        if (!$request->filled('coy') || !$request->filled('bn')) {
            $plRecord = Pl::where('name', $request->pl)->with(['coy', 'coy.bn'])->first();
            if ($plRecord) {
                if (!$request->filled('coy') && $plRecord->coy) {
                    $query->where('coy', $plRecord->coy->name);
                }
                if (!$request->filled('bn') && $plRecord->coy && $plRecord->coy->bn) {
                    $query->where('bn', $plRecord->coy->bn->name);
                }
            }
        }
    }
    if ($request->filled('sec')) {
        $query->where('sec', $request->sec);
        
        // If section is selected but platoon/company/battalion aren't, get the related platoon/company/battalion
        if (!$request->filled('pl') || !$request->filled('coy') || !$request->filled('bn')) {
            $secRecord = Sec::where('name', $request->sec)
                ->with(['pl', 'pl.coy', 'pl.coy.bn'])
                ->first();
                
            if ($secRecord) {
                if (!$request->filled('pl') && $secRecord->pl) {
                    $query->where('pl', $secRecord->pl->name);
                }
                if (!$request->filled('coy') && $secRecord->pl && $secRecord->pl->coy) {
                    $query->where('coy', $secRecord->pl->coy->name);
                }
                if (!$request->filled('bn') && $secRecord->pl && $secRecord->pl->coy && $secRecord->pl->coy->bn) {
                    $query->where('bn', $secRecord->pl->coy->bn->name);
                }
            }
        }
    }
    
    // Sorting with validation
    $allowedSortFields = ['army_number', 'rank', 'name', 'created_at', 'updated_at', 'bn', 'coy', 'pl', 'sec'];
    if ($request->filled('sort') && in_array($request->input('sort'), $allowedSortFields)) {
        $direction = $request->input('direction') === 'desc' ? 'desc' : 'asc';
        $query->orderBy($request->input('sort'), $direction);
    } else {
        // Default sorting
        $query->orderBy('created_at', 'desc');
    }
    
    // Clone query for filtered army numbers before pagination
    $filteredArmyNumbers = (clone $query)->pluck('army_number')->toArray();
    
    // Paginate the results with validation
    $perPage = $request->input('per_page', 25);
    if (!is_numeric($perPage) || $perPage < 1 || $perPage > 100) {
        $perPage = 25; // Default if invalid
    }
    $admissions = $query->paginate($perPage);
    
    // Get filter data efficiently with caching opportunities
    $schools = cache()->remember('schools', 60*60, function() {
        return School::all();
    });
    
    $courses = cache()->remember('courses', 60*60, function() {
        return Course::all();
    });
    
    $ranks = cache()->remember('ranks', 60*60, function() {
        return Admission::distinct()->pluck('rank');
    });
    
    // Get hierarchical filter options
    $bn = Bn::pluck('name');
    $coy = Coy::pluck('name');
    $pl = Pl::pluck('name');
    $sec = Sec::pluck('name');
    
    // Build hierarchy maps for cascade filters
    $hierarchyMaps = $this->buildHierarchyMaps();
    
    // Count total active (non-decommissioned) admissions
    $totalAdmissions = Admission::count(); // This automatically excludes decommissioned records
    
    return view('admissions.index2', compact(
        'admissions',
        'totalAdmissions',
        'schools',
        'courses',
        'ranks',
        'perPage',
        'bn',
        'coy',
        'sec',
        'pl',
        'type',
        'filteredArmyNumbers'
    ) + $hierarchyMaps);
}
    /**
     * Build hierarchical relationship maps for cascade filters
     * 
     * @return array
     */
    private function buildHierarchyMaps()
    {
        // Create mapping for cascade filters
        // Cache these maps for performance since they're unlikely to change frequently
        $companyBnMap = cache()->remember('company_bn_map', 60*10, function() {
            $map = [];
            $coyRecords = Coy::with('bn')->get();
            foreach ($coyRecords as $coyRecord) {
                $map[$coyRecord->name] = $coyRecord->bn->name ?? '';
            }
            return $map;
        });
        
        // Get platoon maps
        list($platoonCoyMap, $platoonBnMap) = cache()->remember('platoon_maps', 60*10, function() {
            $platoonCoyMap = [];
            $platoonBnMap = [];
            $plRecords = Pl::with(['coy', 'coy.bn'])->get();
            
            foreach ($plRecords as $plRecord) {
                $platoonCoyMap[$plRecord->name] = $plRecord->coy->name ?? '';
                $platoonBnMap[$plRecord->name] = $plRecord->coy->bn->name ?? '';
            }
            
            return [$platoonCoyMap, $platoonBnMap];
        });
        
        // Get section maps
        list($sectionPlMap, $sectionCoyMap, $sectionBnMap) = cache()->remember('section_maps', 60*10, function() {
            $sectionPlMap = [];
            $sectionCoyMap = [];
            $sectionBnMap = [];
            
            $secRecords = Sec::with(['pl', 'pl.coy', 'pl.coy.bn'])->get();
            foreach ($secRecords as $secRecord) {
                $sectionPlMap[$secRecord->name] = $secRecord->pl->name ?? '';
                $sectionCoyMap[$secRecord->name] = $secRecord->pl->coy->name ?? '';
                $sectionBnMap[$secRecord->name] = $secRecord->pl->coy->bn->name ?? '';
            }
            
            return [$sectionPlMap, $sectionCoyMap, $sectionBnMap];
        });
        
        return [
            'companyBnMap' => $companyBnMap,
            'platoonCoyMap' => $platoonCoyMap,
            'platoonBnMap' => $platoonBnMap,
            'sectionPlMap' => $sectionPlMap,
            'sectionCoyMap' => $sectionCoyMap,
            'sectionBnMap' => $sectionBnMap
        ];
    }
    
public function index(Request $request)
{
    // Start with base query
    $query = Admission::query();

    // Search functionality
    if ($request->filled('search')) {
        $search = $request->input('search');
        $query->where(function($q) use ($search) {
            $q->where('army_number', 'LIKE', "%{$search}%")
              ->orWhere('rank', 'LIKE', "%{$search}%")
              ->orWhere('name', 'LIKE', "%{$search}%");
        });
    }

    // Additional Filters
    if ($request->filled('school_id')) {
        $query->where('school_id', $request->school_id);
    }
    if ($request->filled('course_id')) {
        $query->where('course_id', $request->course_id);
    }
    if ($request->filled('bn')) {
        $query->where('bn', $request->bn);
    }
    if ($request->filled('coy')) {
        $query->where('coy', $request->coy);
    }
    if ($request->filled('pl')) {
        $query->where('pl', $request->pl);
    }
    if ($request->filled('sec')) {
        $query->where('sec', $request->sec);
    }

    // Get the pagination size from request (default to 25)
    $perPage = $request->input('per_page', 25);

    // Apply eager loading for related models
    $query->with(['school', 'course', 'student']);

    // For debugging - uncomment to check the actual SQL query
    // dd($query->toSql(), $query->getBindings());

    // Apply sorting and pagination
    $admissions = $query->sortable($request)->paginate($perPage);

    $totalAdmissions = Admission::count();
    $schools = School::all();
    $courses = Course::all();
    $ranks = Admission::distinct()->pluck('rank');
    $bn = Bn::pluck('name');
    $coy = Coy::pluck('name');
    $sec = Sec::pluck('name');
    $pl = Pl::pluck('name');

    return view('admissions.index', compact(
        'admissions', 'totalAdmissions', 'schools', 'courses', 
        'ranks', 'perPage', 'bn', 'coy', 'sec', 'pl'
    ));
}


    public function create()
    {
        $schools = School::all();
        $intakes = Intake::all();
        $courses= Course::all();
        $subjects = Subject::all();

        $bn = Bn::pluck('name');
        $coy = Coy::pluck('name');
        $sec = Sec::pluck('name');
        $pl = Pl::pluck('name');



        return view('admissions.create', compact('schools', 'intakes','subjects','courses','bn','coy','sec','pl'));
    }




   
    public function searchArmyNumber2(Request $request)
    {
        $armyNumber = $request->input('army_number');
        $record = ArmyRecord::where('ArmyNumber', $armyNumber)->first();





        if ($record) {
            return response()->json([
                'name' => $record->Student_Name,
                'Rank' => $record->Rank,
                'Student_Name' => $record->Student_Name,
                'Status' => $record->Status,
                'SEX' => $record->SEX,
                'NON_UGANDAN' => $record->NON_UGANDAN,
                'AGE' => $record->AGE,
                'NIN' => $record->NIN,
                'LEVEL_OF_EDUCATION' => $record->LEVEL_OF_EDUCATION,
                'FATHER_S_NAME' => $record->FATHER_S_NAME,
                'NOK' => $record->NOK,
                'NOK_TEL' => $record->NOK_TEL,
                'GUN_NO' => $record->GUN_NO,
                'DISTRICT' => $record->DISTRICT,
                'COUNTY' => $record->COUNTY,
                'S_COUNTY' => $record->S_COUNTY,
                'PARISH' => $record->PARISH,
                'VILLAGE' => $record->VILLAGE,
                'UNIFORM_SIZE' => $record->UNIFORM_SIZE,
                'SHOES_SIZE' => $record->SHOES_SIZE,
                'UNIT' => $record->UNIT,
                'COMBINATION' => $record->COMBINATION,
                'COY' => $record->COY,
                'PL' => $record->PL,
                'SEC' => $record->SEC,
            ]);
        } else {
            return response()->json(['message' => 'Record not found'], 404);
        }



        // return response()->json(['error' => 'No record found'], 404);
    }
/**
 * Search for a record by army number
 * 
 * @param Request $request
 * @return \Illuminate\Http\JsonResponse
 */
public function searchArmyNumber(Request $request)
{
    try {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'army_number' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        // Sanitize the input
        $armyNumber = trim($request->input('army_number'));
        
        // Log the search attempt (optional)
        Log::info('Army number search attempt', [
            'army_number' => $armyNumber,
            'user_id' => auth()->id() ?? 'guest',
            'ip' => $request->ip()
        ]);
        
        // Search for the record
        $record = Admission::where('army_number', $armyNumber)->first();
        
        // Handle not found case
        if (!$record) {
            return response()->json([
                'success' => false,
                'message' => 'No record found for the provided army number'
            ], 404);
        }
        
        // Check if decommissioned
        if ($record->is_decommissioned == 1) {
            return response()->json([
                'success' => false,
                'message' => 'This army number has been decommissioned'
            ], 404);
        }
        
        // Return the data with a consistent response structure
        return response()->json([
            'success' => true,
            'message' => 'Record found',
            'data' => [
                'name' => $record->name,
                'MIL_ID_No' => $record->MIL_ID_No,
                // Add any other required fields here
            ]
        ], 200);
    } catch (\Exception $e) {
        // Log the error for debugging
        Log::error('Error in army number search', [
            'message' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        
        // Don't expose internal error details to the client
        return response()->json([
            'success' => false,
            'message' => 'An unexpected error occurred while processing your request'
        ], 500);
    }
}
    
    // public function getSubjects(Course $course)
    // {
    //     $subjects = $course->subjects()->select('id', 'name')->get();
    //     return response()->json($subjects);
    // }

    public function getSubjects($courseId)
    {
        $subjects = Subject::whereHas('courses', function ($query) use ($courseId) {
            $query->where('course_id', $courseId);
        })->get();

        return response()->json($subjects);
    }

    public function searchRegNumber(Request $request)
    {
        $RegNumber = $request->input('registration_number');
        $record = Registration::where('registration_number', $RegNumber)->first();



        if ($record) {
            return response()->json([
                'name' => $record->name,
                'registration_number' => $record->registration_number
            ]);
        }


        return response()->json(['error' => 'No record found'], 404);
    }

    public function store(Request $request)
    {
        try {
            // Basic validation rules with better error messages
            $validationRules = [
                'name' => ['required', 'string', 'max:255'],
                'rank' => ['required', 'string', 'max:255'],
                'school_id' => ['required', 'exists:schools,id'],
                'course_id' => ['required', 'exists:courses,id'],
                'intake_id' => ['required', 'exists:intakes,id'],
                'subjects' => ['required', 'array', 'min:1'],
                'subjects.*' => ['required', 'integer', 'exists:subjects,id'],
                'email' => ['nullable', 'email:rfc,dns', 'max:255'],
                'phone_number' => ['nullable', 'string', 'regex:/^[+]?[\d\s-]{8,15}$/'],
                'sex' => ['required', 'in:M,F'],
                'bn' => ['nullable', 'string', 'max:255'],
                'coy' => ['nullable', 'string', 'max:255'],
                'pl' => ['nullable', 'string', 'max:255'],
                'sec' => ['nullable', 'string', 'max:255'],
                'image' => ['nullable', 'image'], // 2MB max
                'nin' => ['nullable', 'string', 'max:255', 'unique:admissions,nin'],
                'is_non_ugandan' => ['required', 'in:YES,NO'],
                'army_number' => ['required', 'string', 'max:255', 'unique:admissions,army_number'],
                'home_district' => ['nullable', 'string', 'max:255'],
                'village' => ['nullable', 'string', 'max:255'],
                'father_name' => ['nullable', 'string', 'max:255'],
                'father_phone' => ['nullable', 'string', 'regex:/^[+]?[\d\s-]{8,15}$/'],
                'mother_name' => ['nullable', 'string', 'max:255'],
                'mother_phone' => ['nullable', 'string', 'regex:/^[+]?[\d\s-]{8,15}$/'],
                'spouse_name' => ['nullable', 'string', 'max:255'],
                'spouse_phone' => ['nullable', 'string', 'regex:/^[+]?[\d\s-]{8,15}$/'],
                'education_level' => ['nullable', 'string', 'max:255'],
                'next_kin' => ['nullable', 'string', 'max:255'],
                'next_kin_phone' => ['nullable', 'string'],
                'next_kin_rel' => ['nullable', 'string', 'max:255'],
                'shoe_size' => ['nullable', 'integer', 'min:1', 'max:50'],
                'country_origin' => ['sometimes', 'required_if:is_non_ugandan,YES', 'string', 'max:255'],
            ];

            // Add conditional validation for non-Ugandan status
            if ($request->input('is_non_ugandan') === 'YES') {
                $validationRules['country_origin'] = ['required', 'string', 'max:255'];
            }



            // Custom validation messages
            $customMessages = [
                'name.required' => 'The name field is required.',
                'rank.required' => 'The rank field is required.',
                'school_id.required' => 'Please select a school.',
                'course_id.required' => 'Please select a course.',
                'intake_id.required' => 'Please select an intake.',
                'subjects.required' => 'Please select at least one subject.',
                'subjects.min' => 'Please select at least one subject.',
                'sex.required' => 'Please select a gender.',
                'bn.required' => 'The BN field is required.',
                'coy.required' => 'The COY field is required.',
                'pl.required' => 'The PL field is required.',
                'sec.required' => 'The SEC field is required.',
                'army_number.required' => 'The army number is required.',
                'army_number.unique' => 'This army number is already registered.',
                'is_non_ugandan.required' => 'Please specify whether the cadet is non-Ugandan.',
                'phone_number.regex' => 'Please enter a valid phone number.',
                'father_phone.regex' => 'Please enter a valid phone number for father.',
                'mother_phone.regex' => 'Please enter a valid phone number for mother.',
                'spouse_phone.regex' => 'Please enter a valid phone number for spouse.',
                'image.max' => 'The profile image must not be larger than 2MB.',
                'image.mimes' => 'The profile image must be a JPEG, PNG, or JPG file.',
            ];

            // Log the request data for debugging
            Log::info('Admission store request data:', $request->except(['image']));

            // Validate request
            $validated = $request->validate($validationRules, $customMessages);

            

            DB::beginTransaction();

            try {
                // Handle image upload
                $imagePath = null;
                if ($request->hasFile('image')) {
                    // Generate unique filename with timestamp
                    $filename = 'admission_' . time() . '_' . uniqid() . '.' . $request->image->extension();

                    // Define your custom path
                    $customPath = public_path('admissions/images');

                    // Make sure the directory exists
                    if (!file_exists($customPath)) {
                        mkdir($customPath, 0777, true);
                    }

                    // Move the file to your custom directory
                    $request->file('image')->move($customPath, $filename);

                    // Set the image path relative to public for database storage
                    $imagePath = 'admissions/images/' . $filename;

                    // Optimize image if intervention/image is installed
                    if (class_exists('Image')) {
                        $img = Image::make(public_path($imagePath));
                        $img->resize(800, null, function ($constraint) {
                            $constraint->aspectRatio();
                            $constraint->upsize();
                        });
                        $img->save();
                    }
                }

                // Prepare admission data
                $admissionData = array_merge($validated, [
                    'is_non_ugandan' => $request->input('is_non_ugandan') === 'YES',
                    'year_admission' => now()->year,
                    'created_by' => auth()->id(),
                    'image' => $imagePath,
                ]);

                // Remove subjects from admissionData
                unset($admissionData['subjects']);

                // Create admission
                $admission = Admission::create($admissionData);

                // Attach subjects
                if (!empty($validated['subjects'])) {
                    // Filter out any null values and ensure all values are integers
                    $subjectIds = collect($validated['subjects'])
                        ->filter()
                        ->map(function ($id) {
                            return (int) $id;
                        })
                        ->values()
                        ->all();

                    if (!empty($subjectIds)) {
                        $admission->subjects()->attach($subjectIds);

                        // Log attached subjects for debugging
                        Log::info('Subjects attached:', ['subjects' => $subjectIds]);
                    }
                }

                              // After creating the admission, handle admission phase
                $this->createAdmissionPhase($admission, $validated);

                DB::commit();

   

                // Log success
                Log::info('Admission created successfully', [
                    'admission_id' => $admission->id,
                    'army_number' => $admission->army_number,
                    'created_by' => auth()->id()
                ]);

                return redirect()
                    ->route('admissions.index')
                    ->with('success', 'Admission created successfully');

            } catch (\Exception $e) {
                DB::rollBack();

                // Clean up uploaded image if exists
                if (isset($imagePath) && file_exists(public_path($imagePath))) {
                    unlink(public_path($imagePath));
                }

                Log::error('Database transaction failed', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);

                throw $e;
            }

        } catch (ValidationException $e) {
            Log::error('Validation failed:', [
                'errors' => $e->errors(),
                'request_data' => $request->except(['image'])
            ]);
            return redirect()
                ->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            Log::error('Admission creation failed:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->except(['image'])
            ]);
            return redirect()
                ->back()
                ->withErrors(['error' => 'Failed to create admission: ' . $e->getMessage()])
                ->withInput();
        }
    }


    /**
 * Create admission phase for the given admission
 *
 * @param Admission $admission
 * @param array $validated
 * @return void
 */
protected function createAdmissionPhase(Admission $admission, array $validated)
{
    // Check if any of the phase-related fields are provided
    $phaseFields = ['bn', 'coy', 'pl', 'sec'];
    $phaseData = [];


    $phase = Phase::where('is_active', 1)->first();

    foreach ($phaseFields as $field) {
        if (!empty($validated[$field])) {
            $phaseData[$field] = $validated[$field];
        }
    }

    // If any phase-related data exists, create an admission phase
    if (!empty($phaseData)) {
        AdmissionPhase::create([
            'admission_id' => $admission->id,
            'phase' =>  $phase->name, // First phase by default
            'bn' => $phaseData['bn'] ?? null,
            'coy' => $phaseData['coy'] ?? null,
            'pl' => $phaseData['pl'] ?? null,
            'sec' => $phaseData['sec'] ?? null
        ]);

        // Optional: Log the phase creation
        Log::info('Admission Phase created', [
            'admission_id' => $admission->id,
            'phase_data' => $phaseData
        ]);
    }
}

    

    public function update(Request $request, Admission $admission)
    {
        try {
            // Basic validation rules with better error messages
            $validationRules = [
                'name' => ['nullable', 'string', 'max:255'],
                'rank' => ['nullable', 'string', 'max:255'],
                'school_id' => ['nullable', 'exists:schools,id'],
                'course_id' => ['nullable', 'exists:courses,id'],
                'intake_id' => ['nullable', 'exists:intakes,id'],
                'subjects' => ['nullable', 'array', 'min:1'],
                'subjects.*' => ['nullable', 'integer', 'exists:subjects,id'],
                'email' => ['nullable', 'email:rfc,dns', 'max:255'],
                'phone_number' => ['nullable', 'string', 'regex:/^[+]?[\d\s-]{8,15}$/'],
                'sex' => ['nullable', 'in:M,F'],
                'bn' => ['nullable', 'string', 'max:255'],
                'coy' => ['nullable', 'string', 'max:255'],
                'pl' => ['nullable', 'string', 'max:255'],
                'sec' => ['nullable', 'string', 'max:255'],
                'next_kin_phone' => ['nullable', 'string'],
                'next_kin_rel' => ['nullable', 'string', 'max:255'],
                'image' => ['nullable', 'image'], // 2MB max
                'nin' => [
                    'nullable',
                    'string',
                    'max:255',
                    Rule::unique('admissions', 'nin')->ignore($admission->id)
                ],
                'is_non_ugandan' => ['nullable', 'in:YES,NO'],
                'army_number' => [
                    'nullable',
                    'string',
                    'max:255',
                    Rule::unique('admissions', 'army_number')->ignore($admission->id)
                ],
                'home_district' => ['nullable', 'string', 'max:255'],
                 'country_origin' => ['nullable', 'string', 'max:255'],
                'village' => ['nullable', 'string', 'max:255'],
                'father_name' => ['nullable', 'string', 'max:255'],
                'father_phone' => ['nullable', 'string', 'regex:/^[+]?[\d\s-]{8,15}$/'],
                'mother_name' => ['nullable', 'string', 'max:255'],
                'mother_phone' => ['nullable', 'string', 'regex:/^[+]?[\d\s-]{8,15}$/'],
                'spouse_name' => ['nullable', 'string', 'max:255'],
                'spouse_phone' => ['nullable', 'string', 'regex:/^[+]?[\d\s-]{8,15}$/'],
                'education_level' => ['nullable', 'string', 'max:255'],
                'next_kin' => ['nullable', 'string', 'max:255'],
                'shoe_size' => ['nullable', 'integer', 'min:1', 'max:50'],
            ];

            // Add conditional validation for non-Ugandan status
            if ($request->input('is_non_ugandan') === 'YES') {
                $validationRules['country_origin'] = ['required', 'string', 'max:255'];
            }

            if ($request->input('is_decommissioned')) {
                $request->validate([
                    'decommission_date' => 'required|date',
                    'decommission_reason' => 'required|string|max:255',
                    'decommission_notes' => 'nullable|string',
                ]);
            }

            // Custom validation messages
            $customMessages = [
                'name.required' => 'The name field is required.',
                'rank.required' => 'The rank field is required.',
                'school_id.required' => 'Please select a school.',
                'course_id.required' => 'Please select a course.',
                'intake_id.required' => 'Please select an intake.',
                'subjects.required' => 'Please select at least one subject.',
                'subjects.min' => 'Please select at least one subject.',
                'sex.required' => 'Please select a gender.',
                'bn.required' => 'The BN field is required.',
                'coy.required' => 'The COY field is required.',
                'pl.required' => 'The PL field is required.',
                'sec.required' => 'The SEC field is required.',
                'army_number.required' => 'The army number is required.',
                'army_number.unique' => 'This army number is already registered.',
                'is_non_ugandan.required' => 'Please specify whether the cadet is non-Ugandan.',
                'phone_number.regex' => 'Please enter a valid phone number.',
                'father_phone.regex' => 'Please enter a valid phone number for father.',
                'mother_phone.regex' => 'Please enter a valid phone number for mother.',
                'spouse_phone.regex' => 'Please enter a valid phone number for spouse.',
                'image.max' => 'The profile image must not be larger than 2MB.',
                'image.mimes' => 'The profile image must be a JPEG, PNG, or JPG file.',
            ];

            // Log the request data for debugging
            Log::info('Admission update request data:', $request->except(['image']));



            // Validate request
            $validated = $request->validate($validationRules, $customMessages);

            DB::beginTransaction();

            try {
                // Handle image upload
                $imagePath = $admission->image; // Keep existing image by default
                if ($request->hasFile('image')) {
                    // Delete old image if exists
                    if ($admission->image && file_exists(public_path($admission->image))) {
                        unlink(public_path($admission->image));
                    }

                    // Generate unique filename with timestamp
                    $filename = 'admission_' . time() . '_' . uniqid() . '.' . $request->image->extension();

                    // Define your custom path
                    $customPath = public_path('admissions/images');

                    // Make sure the directory exists
                    if (!file_exists($customPath)) {
                        mkdir($customPath, 0777, true);
                    }

                    // Move the file to your custom directory
                    $request->file('image')->move($customPath, $filename);

                    // Set the image path relative to public for database storage
                    $imagePath = 'admissions/images/' . $filename;

                    // Optimize image if intervention/image is installed
                    if (class_exists('Image')) {
                        $img = Image::make(public_path($imagePath));
                        $img->resize(800, null, function ($constraint) {
                            $constraint->aspectRatio();
                            $constraint->upsize();
                        });
                        $img->save();
                    }
                }

                // Prepare admission data
                $admissionData = array_merge($validated, [
                    'is_non_ugandan' => $request->input('is_non_ugandan') === 'YES',
                    'updated_by' => auth()->id(),
                    'image' => $imagePath,
                ]);

                // Remove subjects from admissionData
                unset($admissionData['subjects']);

                 // Add decommission data if present
            if ($request->input('is_decommissioned')) {
                $admissionData = array_merge($admissionData, [
                    'is_decommissioned' => true,
                    'decommission_date' => $request->input('decommission_date'),
                    'decommission_reason' => $request->input('decommission_reason'),
                    'decommission_notes' => $request->input('decommission_notes'),
                    'decommissioned_by' => auth()->id(),
                ]);
            }

                // Update admission
                $admission->update($admissionData);

                // Sync subjects
                if (!empty($validated['subjects'])) {
                    // Filter out any null values and ensure all values are integers
                    $subjectIds = collect($validated['subjects'])
                        ->filter()
                        ->map(function ($id) {
                            return (int) $id;
                        })
                        ->values()
                        ->all();

                    if (!empty($subjectIds)) {
                        $admission->subjects()->sync($subjectIds);

                        // Log attached subjects for debugging
                        Log::info('Subjects updated:', ['subjects' => $subjectIds]);
                    }
                }

                  $this->updateAdmissionPhase($admission);

                DB::commit();

                // Log success
                Log::info('Admission updated successfully', [
                    'admission_id' => $admission->id,
                    'army_number' => $admission->army_number,
                    'updated_by' => auth()->id()
                ]);

                return redirect()
                    ->route('admissions.index')
                    ->with('success', 'Admission updated successfully');

            } catch (\Exception $e) {
                DB::rollBack();

                // Clean up uploaded image if exists and it's a new image
                if (isset($imagePath) && $imagePath !== $admission->image && file_exists(public_path($imagePath))) {
                    unlink(public_path($imagePath));
                }

                Log::error('Database transaction failed', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);

                throw $e;
            }

        } catch (ValidationException $e) {
            Log::error('Validation failed:', [
                'errors' => $e->errors(),
                'request_data' => $request->except(['image'])
            ]);
            return redirect()
                ->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            Log::error('Admission update failed:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->except(['image'])
            ]);
            return redirect()
                ->back()
                ->withErrors(['error' => 'Failed to update admission: ' . $e->getMessage()])
                ->withInput();
        }
    }


    protected function updateAdmissionPhase(Admission $admission)
{




        // Find the latest phase or create a new one
   $phase = Phase::where('is_active', 1)->first();


        AdmissionPhase::create([
            'admission_id' => $admission->id,
            'phase' =>  $phase->name,
            'bn' =>  $admission->bn ?? null,
            'coy' =>  $admission->coy ?? null,
            'pl' =>  $admission->pl ?? null,
            'sec' =>  $admission->sec ?? null
        ]);


  
}





public function destroy($id)
{
    $admission = Admission::findOrFail($id);
    $admission->delete();

    return redirect()->route('admissions.index')->with('success', 'Admission deleted successfully.');
}



public function edit($id)
{
    $admission = Admission::findOrFail($id);
    $schools = School::all();
    $courses = Course::all();
    $subjects = Subject::all();
    $intakes = Intake::all();
    $bn = Bn::all();
    $coy = Coy::all();
    $pl = Pl::all();
    $sec = Sec::all();

    $phase  = Phase::where('is_active', 1)->first();

    return view('admissions.edit', compact('admission','phase', 'schools', 'courses', 'subjects', 'intakes','bn','coy','pl','sec'));
}







public function show($id)
{
    $admission = Admission::with(['school', 'course', 'intake', 'subjects'])->findOrFail($id);
    return view('admissions.show', compact('admission'));
}



}
