<?php



namespace App\Http\Controllers;

use App\Imports\AdmissionsImport;
use App\Models\School;
use App\Models\Course;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Validators\ValidationException;
use Illuminate\Support\Facades\Validator;

class AdmissionsImportController extends Controller
{
    public function import(Request $request)
    {
        // 1. Initial request validation
        $validator = Validator::make($request->all(), [
            'file' => [
                'required',
                'file',
                'mimes:xlsx,xls,csv',
                'max:10240', // 10MB max file size
            ]
        ]);

        if ($validator->fails()) {
            return back()
                ->withErrors($validator)
                ->withInput();
        }

        // 2. File handling and backup
        try {
            // Store original file for backup
            $originalName = $request->file('file')->getClientOriginalName();
            $timestamp = now()->format('Y-m-d_H-i-s');
            $backupPath = $request->file('file')->storeAs(
                'imports/admissions/backup',
                "{$timestamp}_{$originalName}"
            );

            // 3. Pre-import validation
            $this->validatePrerequisites();

            // 4. Process import
            DB::beginTransaction();
            
            try {
                $import = new AdmissionsImport;
                
                // Track start time for performance logging
                $startTime = microtime(true);
                
                Excel::import($import, $request->file('file'));
                
                $executionTime = microtime(true) - $startTime;

                // Check for any errors during import
                $errors = $import->getErrors();
                if (!empty($errors)) {
                    throw new \Exception(implode("\n", $errors));
                }

                // 5. Log successful import
                $this->logImportSuccess($import, $originalName, $executionTime);

                DB::commit();

                // 6. Return success response with statistics
                $stats = $this->getImportStats($import);
                return back()->with([
                    'success' => 'Admissions imported successfully.',
                    'stats' => $stats
                ]);

            } catch (ValidationException $e) {
                DB::rollBack();
                $this->logImportError($e, $originalName);
                
                return back()->with('error', [
                    'message' => 'Validation failed during import',
                    'details' => $this->formatValidationErrors($e->failures())
                ]);

            } catch (\Exception $e) {
                DB::rollBack();
                $this->logImportError($e, $originalName);

                return back()->with('error', [
                    'message' => 'Error during import process',
                    'details' => $this->formatErrorMessage($e->getMessage())
                ]);
            }

        } catch (\Exception $e) {
            Log::error('File handling error during admission import', [
                'error' => $e->getMessage(),
                'file' => $originalName ?? 'unknown'
            ]);

            return back()->with('error', 'Unable to process the uploaded file.');
        }
    }

    private function validatePrerequisites()
    {
        // Verify essential data exists
        $schoolCount = School::count();
        $courseCount = Course::count();

        if ($schoolCount === 0) {
            throw new \Exception('No schools found in the system. Please add schools before importing admissions.');
        }

        if ($courseCount === 0) {
            throw new \Exception('No courses found in the system. Please add courses before importing admissions.');
        }
    }

    private function logImportSuccess($import, $filename, $executionTime)
    {
        Log::info('Admission import completed successfully', [
            'filename' => $filename,
            'execution_time' => round($executionTime, 2) . ' seconds',
            'rows_processed' => $import->getRowCount() ?? 0,
            'user_id' => auth()->id(),
            'timestamp' => now()->toDateTimeString()
        ]);
    }

    private function logImportError($exception, $filename)
    {
        Log::error('Error during admission import', [
            'filename' => $filename,
            'error' => $exception->getMessage(),
            'user_id' => auth()->id(),
            'timestamp' => now()->toDateTimeString(),
            'trace' => $exception->getTraceAsString()
        ]);
    }

    private function formatValidationErrors($failures)
    {
        return collect($failures)->map(function ($failure) {
            return "Row {$failure->row()}: " . implode(', ', $failure->errors());
        })->toArray();
    }

    private function formatErrorMessage($message)
    {
        // Clean up error message for display
        $message = preg_replace('/\s+/', ' ', $message);
        return strlen($message) > 200 ? substr($message, 0, 200) . '...' : $message;
    }

    private function getImportStats($import)
    {
        return [
            'total_processed' => $import->getRowCount() ?? 0,
            'successful' => $import->getSuccessCount() ?? 0,
            'failed' => $import->getFailureCount() ?? 0,
            'processing_time' => $import->getExecutionTime() ?? 0
        ];
    }


    public function showExcel()
    {
        return view('admissions.import');
    }
}
