<?php

namespace App\Http\Controllers;

use App\Models\Admission;
use App\Models\CRAssessment2;
use App\Models\Phase;
use Illuminate\Http\Request;
use PDF;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\DB;

class CRAssessment2Controller extends Controller
{


public function importCRs(Request $request)
{
    $request->validate([
        'import_file' => 'required|file|mimes:xlsx,xls,csv',
        'phase_id' => 'required|exists:phases,id'
    ]);

    try {
        $phaseId = $request->input('phase_id');
        $file = $request->file('import_file');
        
        $data = Excel::toArray([], $file);
        
        if (empty($data)) {
            return redirect()->back()->with('error', 'The uploaded file is empty.');
        }

        $rows = $data[0];
        
        // Find the header row (look for "S/NO" in any row)
        $headerRowIndex = null;
        $expectedHeaders = ['S/NO', 'ARMY NO.', 'RANK', 'FULL NAMES', '22CR'];
        
        foreach ($rows as $index => $row) {
            $rowValues = array_map('strtoupper', array_map('trim', $row));
            if (in_array('S/NO', $rowValues) && in_array('ARMY NO.', $rowValues)) {
                $headerRowIndex = $index;
                break;
            }
        }

        if ($headerRowIndex === null) {
            return redirect()->back()->with('error', 'Could not find header row with columns: S/NO, ARMY NO., RANK, FULL NAMES, 22CR');
        }

        // Get the header and rows after it
        $header = array_map('strtoupper', array_map('trim', $rows[$headerRowIndex]));
        $dataRows = array_slice($rows, $headerRowIndex + 1);
        
        // Map column indexes
        $columnIndexes = [];
        foreach ($expectedHeaders as $expectedHeader) {
            $columnIndexes[$expectedHeader] = array_search($expectedHeader, $header);
        }

        // Validate that all required columns are found
        foreach ($expectedHeaders as $expectedHeader) {
            if ($columnIndexes[$expectedHeader] === false) {
                return redirect()->back()->with('error', "Required column not found: {$expectedHeader}");
            }
        }

        $importedCount = 0;
        $updatedCount = 0;
        $errors = [];

        DB::beginTransaction();

        foreach ($dataRows as $index => $row) {
            $lineNumber = $headerRowIndex + $index + 2; // +2 for header row and 1-based indexing
            
            try {
                // Skip empty rows
                if (empty(array_filter($row))) {
                    continue;
                }

                // Get values using column indexes
                $armyNumber = trim($row[$columnIndexes['ARMY NO.']] ?? '');
                $score = trim($row[$columnIndexes['22CR']] ?? '');
                $rank = trim($row[$columnIndexes['RANK']] ?? '');
                $fullName = trim($row[$columnIndexes['FULL NAMES']] ?? '');

                // Validate required fields
                if (empty($armyNumber)) {
                    $errors[] = "Row {$lineNumber}: Army number is required";
                    continue;
                }

                if (empty($score)) {
                    $errors[] = "Row {$lineNumber}: CR score is required";
                    continue;
                }

                // Validate score is numeric
                if (!is_numeric($score)) {
                    $errors[] = "Row {$lineNumber}: CR score must be numeric (found: '{$score}')";
                    continue;
                }

                $score = floatval($score);

                // Find admission by army number
                $admission = Admission::where('army_number', $armyNumber)->first();

                if (!$admission) {
                    $errors[] = "Row {$lineNumber}: Admission not found for army number: {$armyNumber}";
                    continue;
                }

                // Check if CR assessment already exists for this phase
                $existingAssessment = CRAssessment2::where('admission_id', $admission->id)
                    ->where('phase_id', $phaseId)
                    ->first();

                if ($existingAssessment) {
                    // Update existing record
                    $existingAssessment->update([
                        'score' => $score,
                        'remarks' => $this->calculateRemarks($score),
                        'marked_by' => auth()->user()->name ?? 'System (Import)',
                        'updated_at' => now()
                    ]);
                    $updatedCount++;
                } else {
                    // Create new record
                    CRAssessment2::create([
                        'army_number' => $admission->army_number,
                        'name' => $admission->name,
                        'rank' => $admission->rank,
                        'score' => $score,
                        'remarks' => $this->calculateRemarks($score),
                        'admission_id' => $admission->id,
                        'phase_id' => $phaseId,
                        'marked_by' => auth()->user()->name ?? 'System (Import)',
                        'created_at' => now(),
                        'updated_at' => now()
                    ]);
                    $importedCount++;
                }

            } catch (\Exception $e) {
                $errors[] = "Row {$lineNumber}: " . $e->getMessage();
                continue;
            }
        }

        DB::commit();

        $message = "Import completed successfully!";
        $message .= " New records: {$importedCount}, Updated records: {$updatedCount}";
        
        if (!empty($errors)) {
            $message .= ". Errors: " . count($errors);
            session()->flash('import_errors', $errors);
        }

        return redirect()->route('cr.mark')->with('success', $message);

    } catch (\Exception $e) {
        DB::rollBack();
        return redirect()->back()->with('error', 'Import failed: ' . $e->getMessage());
    }
}

    public function showImportForm()
    {
        $phases = Phase::all();
        $activePhase = Phase::where('is_active', 1)->first();
        
        return view('cr.import', compact('phases', 'activePhase'));
    }

    public function markCRs()
    {


        // Get filter parameters
        $filters = [
            'bn' => request('bn'),
            'coy' => request('coy'),
            'pl' => request('pl'),
            'sec' => request('sec'),
            'marked' => request('marked'),
            'phase' => request('phase'),
        ];

        // Get selected phase or active phase
        $selectedPhaseId = request('phase');
        $activePhase = Phase::where('is_active', 1)->first();
        
        if (!$selectedPhaseId && $activePhase) {
            $selectedPhaseId = $activePhase->id;
        }

        // Query admissions with optional filters
        $query = Admission::query();
        
        if (!empty($filters['bn'])) {
            $query->where('bn', $filters['bn']);
        }
        
        if (!empty($filters['coy'])) {
            $query->where('coy', $filters['coy']);
        }
        
        if (!empty($filters['pl'])) {
            $query->where('pl', $filters['pl']);
        }
        
        if (!empty($filters['sec'])) {
            $query->where('sec', $filters['sec']);
        }

        $admissions = $query->orderBy('army_number')->get();

        // Get existing CR records to check marked status for the selected phase
        $existingCRs = CRAssessment2::whereIn('admission_id', $admissions->pluck('id'))
            ->when($selectedPhaseId, function($query) use ($selectedPhaseId) {
                $query->where('phase_id', $selectedPhaseId);
            })
            ->get()
            ->keyBy('admission_id');

        // Apply marked filter if specified
        if (!empty($filters['marked'])) {
            if ($filters['marked'] == '1') {
                $admissions = $admissions->filter(function($admission) use ($existingCRs) {
                    return isset($existingCRs[$admission->id]);
                });
            } elseif ($filters['marked'] == '0') {
                $admissions = $admissions->filter(function($admission) use ($existingCRs) {
                    return !isset($existingCRs[$admission->id]);
                });
            }
        }

        // Get filter values
        $filterValues = $this->getFilterValues();
        $phases = Phase::all();

        return view('cr.mark', array_merge(compact('admissions', 'filters', 'existingCRs', 'phases', 'selectedPhaseId', 'activePhase'), $filterValues));
    }

    public function updateCRs(Request $request)
    {
        $scores = $request->input('scores', []);
        $phaseId = $request->input('phase_id');
        
        foreach ($scores as $admissionId => $scoreData) {
            if (isset($scoreData['score'])) {
                $crAssessment = CRAssessment2::where('admission_id', $admissionId)
                    ->where('phase_id', $phaseId)
                    ->first();
                
                if ($crAssessment) {
                    // Update existing CR record
                    $crAssessment->update([
                        'score' => $scoreData['score'],
                        'remarks' => $this->calculateRemarks($scoreData['score']),
                        'marked_by' => auth()->user()->name ?? 'System',
                        'updated_at' => now()
                    ]);
                } else {
                    // Create new CR record
                    $admission = Admission::find($admissionId);
                    if ($admission) {
                        CRAssessment2::create([
                            'army_number' => $admission->army_number,
                            'name' => $admission->name,
                            'rank' => $admission->rank,
                            'score' => $scoreData['score'],
                            'remarks' => $this->calculateRemarks($scoreData['score']),
                            'admission_id' => $admissionId,
                            'phase_id' => $phaseId,
                            'marked_by' => auth()->user()->name ?? 'System',
                            'created_at' => now(),
                            'updated_at' => now()
                        ]);
                    }
                }
            }
        }

        return redirect()->route('cr.mark')->with('success', 'CR scores saved successfully!');
    }

    public function overview()
    {


        // Get filter parameters
        $filters = [
            'bn' => request('bn'),
            'coy' => request('coy'),
            'pl' => request('pl'),
            'sort' => request('sort', 'default'),
        ];

        // Query CR assessments with filters
        $query = CRAssessment2::with(['admission', 'phase']);

        if (!empty($filters['bn'])) {
            $query->whereHas('admission', function ($q) use ($filters) {
                $q->where('bn', $filters['bn']);
            });
        }

        if (!empty($filters['coy'])) {
            $query->whereHas('admission', function ($q) use ($filters) {
                $q->where('coy', $filters['coy']);
            });
        }

        if (!empty($filters['pl'])) {
            $query->whereHas('admission', function ($q) use ($filters) {
                $q->where('pl', $filters['pl']);
            });
        }

        // Apply sorting
        switch ($filters['sort']) {
            case 'score_asc':
                $query->orderBy('score', 'asc');
                break;
            case 'score_desc':
                $query->orderBy('score', 'desc');
                break;
            case 'name_asc':
                $query->orderBy('name', 'asc');
                break;
            case 'name_desc':
                $query->orderBy('name', 'desc');
                break;
            default:
                $query->orderBy('army_number', 'asc');
                break;
        }

        $assessments = $query->get();

        // Get filter values
        $filterValues = $this->getFilterValues();

        return view('cr.overview', array_merge(compact('assessments', 'filters'), $filterValues));
    }

    public function exportPDF()
    {
        $filters = [
            'bn' => request('bn'),
            'coy' => request('coy'),
            'pl' => request('pl'),
        ];

        $query = CRAssessment2::with(['admission', 'phase']);

        foreach ($filters as $key => $value) {
            if (!empty($value)) {
                $query->whereHas('admission', function ($q) use ($key, $value) {
                    $q->where($key, $value);
                });
            }
        }

        $assessments = $query->orderBy('army_number', 'asc')->get();

        $pdf = PDF::loadView('cr.pdf', compact('assessments', 'filters'));
        
        return $pdf->download('cr-assessment-overview-' . date('Y-m-d') . '.pdf');
    }

    public function getFilterValues()
    {
        return [
            'bns' => Admission::distinct()->whereNotNull('bn')->pluck('bn')->toArray(),
            'coys' => Admission::distinct()->whereNotNull('coy')->pluck('coy')->toArray(),
            'pls' => Admission::distinct()->whereNotNull('pl')->pluck('pl')->toArray(),
            'secs' => Admission::distinct()->whereNotNull('sec')->pluck('sec')->toArray(),
        ];
    }

    public function calculateRemarks($score)
    {
        if (!$score) return '';

        $numericScore = floatval($score);
        
        if ($numericScore >= 97.2) return '9.72 - EXCELLENT';
        if ($numericScore >= 90) return '9.0+ - VERY GOOD';
        if ($numericScore >= 80) return '8.0+ - GOOD';
        if ($numericScore >= 70) return '7.0+ - AVERAGE';
        if ($numericScore >= 60) return '6.0+ - BELOW AVERAGE';
        return 'FAIL';
    }

    public function getScorePercentage($score)
    {
        if (!$score) return '0%';
        return number_format($score, 2) . '% /20%';
    }
}