<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\CRAssessment;
use App\Models\TwentyTwoCR;
use App\Models\CRAssessmentDetail;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Models\Admission;
use App\Models\School;
use App\Models\Course;
use App\Models\Bn;
use App\Models\Coy;
use App\Models\Sec;
use App\Models\Pl;
use App\Models\Phase;

class CRAssessmentController extends Controller
{


    
    public function store(Request $request)
    {
        
        try {
            // First validate basic requirements
            $validatedData = $request->validate([
                'army_number' => 'required|string',
                'name' => 'required|string|max:255',
                'school_id' => 'required|exists:schools,id',
                'course_id' => 'required|exists:courses,id',
                'intake_id' => 'required|exists:intakes,id',
                'phase' => 'required|string|in:Phase 1,Phase 2,Phase 3',
                'assessments' => 'required|array|min:1'
            ]);
    
            // Validate each assessment individually
            foreach ($request->assessments as $assessment) {
                if ($assessment['rating'] < 1 || $assessment['rating'] > 5 || !is_int((int)$assessment['rating']) || (int)$assessment['rating'] != $assessment['rating']) {
                    return response()->json([
                        'success' => false,
                        'message' => "Invalid rating for CR {$assessment['cr_id']}: must be an integer between 1 and 5"
                    ], 422);
                }
    
                // Validate CR exists and rating is valid
                if (!TwentyTwoCR::find($assessment['cr_id'])) {
                    return response()->json([
                        'success' => false,
                        'message' => "Invalid CR ID: {$assessment['cr_id']}"
                    ], 422);
                }
    
                if ($assessment['rating'] < 1 || $assessment['rating'] > 5) {
                    return response()->json([
                        'success' => false,
                        'message' => "Invalid rating for CR {$assessment['cr_id']}: must be between 1 and 5"
                    ], 422);
                }
            }
    
            \Log::info('Request data:', $request->all());

            if (!$request->has('assessments') || !is_array($request->assessments)) {
                return response()->json([
                    'success' => false,
                    'message' => 'The assessments field is required and must be an array'
                ], 422);
            }
            DB::beginTransaction();
    
            // Create or update assessment
            $assessment = CRAssessment::firstOrCreate(
                [
                    'army_number' => $validatedData['army_number'],
                    'phase' => $validatedData['phase']
                ],
                [
                    'name' => $validatedData['name'],
                    'school_id' => $validatedData['school_id'],
                    'course_id' => $validatedData['course_id'],
                    'intake_id' => $validatedData['intake_id'],
                    'assessment_type' => 'Character Assessment',
                    'assessment_group' => '22 CR'
                ]
            );
    
            // Process each assessment
            foreach ($request->assessments as $data) {
                $detail = CRAssessmentDetail::firstOrNew([
                    'cr_assessment_id' => $assessment->id,
                    'twenty_two_c_r_s_id' => $data['cr_id']
                ]);
    
                $fColumn = "f{$data['rating']}";
                
                if ($detail->exists) {
                    $detail->increment($fColumn);
                    
                    // Recalculate totals
                    $totalF = $detail->f1 + $detail->f2 + $detail->f3 + $detail->f4 + $detail->f5;
                    $weightedSum = ($detail->f1 * 1) + ($detail->f2 * 2) + ($detail->f3 * 3) + 
                                 ($detail->f4 * 4) + ($detail->f5 * 5);
                    
                    $detail->update([
                        'rating' => $totalF > 0 ? round($weightedSum / $totalF) : $data['rating'],
                        'frequency' => $totalF
                    ]);
                } else {
                    $detail->fill([
                        'rating' => $data['rating'],
                        'frequency' => 1,
                        $fColumn => 1
                    ])->save();
                }
            }
    
            DB::commit();
            
            return response()->json([
                'success' => true,
                'message' => 'Assessment updated successfully',
                'data' => [
                    'assessment_id' => $assessment->id,
                    'processed_crs' => collect($request->assessments)->pluck('cr_id')
                ]
            ]);
    
        } catch (\Exception $e) {
            DB::rollback();
            \Log::error('CR Assessment Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error saving assessment: ' . $e->getMessage()
            ], 500);
        }
    }


    public function getAllByArmyNumber(Request $request)
{
    $assessments = CRAssessment::where('army_number', $request->army_number)
        ->with('twentyTwoCR') // Include the relationship
        ->orderBy('created_at', 'desc')
        ->get();



    return response()->json([
        'assessments' => $assessments
    ]);
}



    public function checkAssessment(Request $request)
{
    $assessment = CRAssessment::where('army_number', $request->army_number)
                             ->where('phase', $request->phase)
                             ->first();

    return response()->json([
        'exists' => !is_null($assessment),
        'assessment' => $assessment
    ]);
}


public function printPDF(Request $request)
{
    $data = $request->all();
    
    $pdf = PDF::loadView('cr-assessments.print', [
        'student' => $data['student'],
        'assessments' => $data['assessments'],
        'totalCRES' => $data['totalCRES'],
        'finalScore' => $data['finalScore'],
        'date' => Carbon::parse($data['date'])->format('d/m/Y'),
    ]);

    return $pdf->stream('cr-assessment.pdf');
}

public function show22CR(Request $request)
{
    $assessments = collect();




    
    if ($request->has('army_number')) {
        $armyNumber = $request->army_number;
    
        $assessments = CRAssessment::where('army_number', $armyNumber)

            ->with(['assessmentDetails.twentyTwoCR'])
            ->orderBy('created_at', 'desc')
            ->get()
            ->groupBy('army_number');
    }
    
    
    
    return view('22cr.view', compact('assessments'));
}


// public function overView22CR(Request $request)
// {
//     // Get filter parameters
//     $platoon = $request->input('pl');
//     $sortBy = $request->input('sort', 'default'); // 'best', 'worst', or 'default'
    
//     // Base query
//     $query = CRAssessment::with(['assessmentDetails.twentyTwoCR'])
//         ->join('admissions', 'c_r_assessments.army_number', '=', 'admissions.army_number')
//         ->select('c_r_assessments.*', 'admissions.pl');
    
//     // Apply platoon filter if provided
//     if ($platoon) {
//         $query->where('admissions.pl', $platoon);
//     }
    
//     // Get the results
//     $assessmentResults = $query->orderBy('c_r_assessments.created_at', 'desc')->get();
    
//     // Process results to calculate scores by platoon
//     $platoonScores = [];
//     foreach ($assessmentResults as $assessment) {
//         $totalCRES = 0;
        
//         foreach ($assessment->assessmentDetails as $detail) {
//             // Calculate individual f and fx
//             $f = $detail->f1 + $detail->f2 + $detail->f3 + $detail->f4 + $detail->f5;
//             $fx = ($detail->f1 * 1) + ($detail->f2 * 2) + ($detail->f3 * 3) +
//                   ($detail->f4 * 4) + ($detail->f5 * 5);
            
//             // Calculate individual CRES
//             if ($f > 0) {
//                 $individualCRES = ($fx / $f) * $detail->twentyTwoCR->weight;
//                 $totalCRES += $individualCRES;
//             }
//         }
        
//         $score = ($totalCRES / 720) * 20;
        
//         // Store score by platoon
//         if (!isset($platoonScores[$assessment->pl])) {
//             $platoonScores[$assessment->pl] = [
//                 'count' => 0,
//                 'total' => 0,
//                 'assessments' => []
//             ];
//         }
        
//         $platoonScores[$assessment->pl]['count']++;
//         $platoonScores[$assessment->pl]['total'] += $score;
//         $platoonScores[$assessment->pl]['assessments'][] = $assessment;
//     }
    
//     // Calculate average score for each platoon
//     foreach ($platoonScores as $pl => $data) {
//         $platoonScores[$pl]['average'] = $data['total'] / $data['count'];
//     }
    
//     // Sort platoons by average score
//     if ($sortBy == 'best') {
//         uasort($platoonScores, function($a, $b) {
//             return $b['average'] <=> $a['average'];
//         });
//     } elseif ($sortBy == 'worst') {
//         uasort($platoonScores, function($a, $b) {
//             return $a['average'] <=> $b['average'];
//         });
//     }
    
//     // Group assessments by army number for display
//     $assessments = collect();
//     foreach ($platoonScores as $plData) {
//         foreach ($plData['assessments'] as $assessment) {
//             if (!$assessments->has($assessment->army_number)) {
//                 $assessments[$assessment->army_number] = collect();
//             }
//             $assessments[$assessment->army_number]->push($assessment);
//         }
//     }
    
//     // Get list of platoons for the filter dropdown
//     $platoons = DB::table('admissions')->select('pl')->distinct()->whereNotNull('pl')->orderBy('pl')->pluck('pl');
    
//     // Determine best and worst platoons
//     $bestPlatoon = null;
//     $worstPlatoon = null;
    
//     if (!empty($platoonScores)) {
//         $bestPlatoon = array_keys($platoonScores)[0];
//         $worstPlatoon = array_keys($platoonScores)[count($platoonScores) - 1];
//     }
    
//     return view('22cr.overview', compact('assessments', 'platoons', 'platoon', 'sortBy', 'bestPlatoon', 'worstPlatoon', 'platoonScores'));
// }
public function overView22CR(Request $request)
{
    // Get filter parameters
    $platoon = $request->input('pl');
    $coys = $request->input('coy');
    $section = $request->input('sec');
    $bn = $request->input('bn');
    $schoolId = $request->input('school_id');
    $courseId = $request->input('course_id');
    $intakeId = $request->input('intake_id');
    $yearAdmission = $request->input('year_admission');
    $studentId = $request->input('student_id');
    $sortBy = $request->input('sort', 'default'); // 'asc', 'desc', 'score_asc', 'score_desc' or 'default'
    
    
    // Base query - using the correct table name c_r_assessments


    $query = CRAssessment::with(['assessmentDetails.twentyTwoCR'])
        ->join('admissions', 'c_r_assessments.army_number', '=', 'admissions.army_number')
        ->select(
            'c_r_assessments.*',
            'admissions.pl',
            'admissions.name',
            'admissions.sec',
            'admissions.bn',
            'admissions.coy',
            'admissions.school_id',
            'admissions.course_id',
            'admissions.intake_id',
            'admissions.year_admission',
            'admissions.student_id'
        ); 



    
    // Apply filters if provided
    if ($platoon) {
        $query->where('admissions.pl', $platoon);
    }
    
    if ($coys) {
        $query->where('admissions.coy', $coys);
    }
    
    if ($section) {
        $query->where('admissions.sec', $section);
    }
    
    if ($bn) {
        $query->where('admissions.bn', $bn);
    }
    
    if ($schoolId) {
        $query->where('admissions.school_id', $schoolId);
    }
    
    if ($courseId) {
        $query->where('admissions.course_id', $courseId);
    }
    
    if ($intakeId) {
        $query->where('admissions.intake_id', $intakeId);
    }
    
    if ($yearAdmission) {
        $query->where('admissions.year_admission', $yearAdmission);
    }
    
    if ($studentId) {
        $query->where('admissions.student_id', $studentId);
    }
    
    // Apply initial sorting to the query before getting results
    if ($sortBy === 'asc') {
        $query->orderBy('admissions.name', 'asc');
    } elseif ($sortBy === 'desc') {
        $query->orderBy('admissions.name', 'desc');
    } else {
        // Default sorting by created_at
        $query->orderBy('c_r_assessments.created_at', 'desc');
    }
    
    // Get the results
    $assessmentResults = $query->get();
    
    // Process results to calculate scores by platoon and by student
    $platoonScores = [];
    $studentScores = [];
    
    foreach ($assessmentResults as $assessment) {
        $totalCRES = 0;
        
        foreach ($assessment->assessmentDetails as $detail) {
            // Calculate individual f and fx
            $f = $detail->f1 + $detail->f2 + $detail->f3 + $detail->f4 + $detail->f5;
            $fx = ($detail->f1 * 1) + ($detail->f2 * 2) + ($detail->f3 * 3) +
                  ($detail->f4 * 4) + ($detail->f5 * 5);
            
            // Calculate individual CRES
            if ($f > 0) {
                $individualCRES = ($fx / $f) * $detail->twentyTwoCR->weight;
                $totalCRES += $individualCRES;
            }
        }
        
        $score = ($totalCRES / 720) * 20;
        
        // Skip if platoon is null
        if (!$assessment->pl) {
            continue;
        }
        
        // Store score by platoon
        if (!isset($platoonScores[$assessment->pl])) {
            $platoonScores[$assessment->pl] = [
                'count' => 0,
                'total' => 0,
                'assessments' => [],
                'bestStudent' => [
                    'army_number' => '',
                    'name' => '',
                    'score' => 0
                ],
                'worstStudent' => [
                    'army_number' => '',
                    'name' => '',
                    'score' => 20 // Start with max score
                ]
            ];
        }
        
        $platoonScores[$assessment->pl]['count']++;
        $platoonScores[$assessment->pl]['total'] += $score;
        $platoonScores[$assessment->pl]['assessments'][] = $assessment;
        
        // Store score by student
        if (!isset($studentScores[$assessment->army_number])) {
            $studentScores[$assessment->army_number] = [
                'name' => $assessment->name,
                'pl' => $assessment->pl,
                'scores' => [],
                'total' => 0,
                'count' => 0
            ];
        }
        
        $studentScores[$assessment->army_number]['scores'][] = $score;
        $studentScores[$assessment->army_number]['total'] += $score;
        $studentScores[$assessment->army_number]['count']++;
    }
    
    // Calculate average score for each platoon and find best/worst students
    foreach ($platoonScores as $pl => $data) {
        $platoonScores[$pl]['average'] = $data['total'] / $data['count'];
    }
    
    // Calculate average score for each student
    foreach ($studentScores as $armyNumber => &$studentData) {
        $studentData['average'] = $studentData['total'] / $studentData['count'];
        
        // Update best/worst student for this platoon
        $pl = $studentData['pl'];
        if (isset($platoonScores[$pl])) {
            // Check if this is the best student
            if ($studentData['average'] > $platoonScores[$pl]['bestStudent']['score']) {
                $platoonScores[$pl]['bestStudent'] = [
                    'army_number' => $armyNumber,
                    'name' => $studentData['name'],
                    'score' => $studentData['average']
                ];
            }
            
            // Check if this is the worst student
            if ($studentData['average'] < $platoonScores[$pl]['worstStudent']['score']) {
                $platoonScores[$pl]['worstStudent'] = [
                    'army_number' => $armyNumber,
                    'name' => $studentData['name'],
                    'score' => $studentData['average']
                ];
            }
        }
    }
    
    // Sort platoons by average score for "best" and "worst" options
    if ($sortBy == 'best') {
        uasort($platoonScores, function($a, $b) {
            return $b['average'] <=> $a['average'];
        });
    } elseif ($sortBy == 'worst') {
        uasort($platoonScores, function($a, $b) {
            return $a['average'] <=> $b['average'];
        });
    }
    
    // Group assessments by army number for display
    $assessments = collect();
    foreach ($platoonScores as $plData) {
        foreach ($plData['assessments'] as $assessment) {
            if (!$assessments->has($assessment->army_number)) {
                $assessments[$assessment->army_number] = collect();
            }
            $assessments[$assessment->army_number]->push($assessment);
        }
    }
    
    // Handle score sorting if needed (for score_asc, score_desc)
    if ($sortBy === 'score_asc' || $sortBy === 'score_desc') {
        // Sort by scores
        $sortedAssessments = collect();
        $tempScoreMap = [];
        
        // Calculate average score for each army number
        foreach ($assessments as $armyNumber => $armyAssessments) {
            $totalScore = 0;
            $count = 0;
            
            foreach ($armyAssessments as $assessment) {
                $assessmentScore = 0;
                foreach ($assessment->assessmentDetails as $detail) {
                    $f = $detail->f1 + $detail->f2 + $detail->f3 + $detail->f4 + $detail->f5;
                    $fx = ($detail->f1 * 1) + ($detail->f2 * 2) + ($detail->f3 * 3) +
                          ($detail->f4 * 4) + ($detail->f5 * 5);
                    
                    if ($f > 0) {
                        $individualCRES = ($fx / $f) * $detail->twentyTwoCR->weight;
                        $assessmentScore += $individualCRES;
                    }
                }
                $score = ($assessmentScore / 720) * 20;
                $totalScore += $score;
                $count++;
            }
            
            $avgScore = $count > 0 ? $totalScore / $count : 0;
            $tempScoreMap[$armyNumber] = $avgScore;
        }
        
        // Sort by score
        arsort($tempScoreMap); // High to low
        if ($sortBy === 'score_asc') {
            $tempScoreMap = array_reverse($tempScoreMap, true); // Low to high
        }
        
        // Rebuild the assessments collection in the right order
        foreach ($tempScoreMap as $armyNumber => $score) {
            $sortedAssessments[$armyNumber] = $assessments[$armyNumber];
        }
        
        $assessments = $sortedAssessments;
    }

    // Get lists for filter dropdowns
    $intakes = DB::table('admissions')->select('intake_id')->distinct()->whereNotNull('intake_id')->orderBy('intake_id')->pluck('intake_id');
    $years = DB::table('admissions')->select('year_admission')->distinct()->whereNotNull('year_admission')->orderBy('year_admission')->pluck('year_admission');
    
    $schools = School::all();
    $courses = Course::all();
    $ranks = Admission::distinct()->pluck('rank');
    $battalions = Bn::pluck('name');
    $coysList = Coy::pluck('name');
    $sections = Sec::pluck('name');
    $platoons = Pl::pluck('name');

    // Determine best and worst platoons
    $bestPlatoon = null;
    $worstPlatoon = null;
    
    if (!empty($platoonScores)) {
        $bestPlatoon = array_keys($platoonScores)[0];
        $worstPlatoon = array_keys($platoonScores)[count($platoonScores) - 1];
    }
    
    return view('22cr.overview', compact(
        'assessments', 
        'platoons',
        'coysList',
        'sections',
        'battalions',
        'schools',
        'courses',
        'intakes',
        'years',
        'platoon',
        'coys',
        'section',
        'bn',
        'schoolId',
        'courseId',
        'intakeId',
        'yearAdmission',
        'studentId',
        'sortBy', 
        'bestPlatoon', 
        'worstPlatoon', 
        'platoonScores',
        'studentScores'
    ));
}
public function getDetails(CRAssessment $assessment)
{


    return response()->json($assessment->load('assessmentDetails.twentyTwoCR'));
}


public function generatePDF(Request $request)
{

    // Fetching the input filters
    $platoon = $request->input('pl');
    $coys = $request->input('coy');
    $section = $request->input('sec');
    $bn = $request->input('bn');
    $schoolId = $request->input('school_id');
    $courseId = $request->input('course_id');
    $intakeId = $request->input('intake_id');
    $yearAdmission = $request->input('year_admission');
    $studentId = $request->input('student_id');
    $sortBy = $request->input('sort', 'default'); // 'best', 'worst', or 'default'

    // Base query - join with admissions and apply filters
    // Base query - using the correct table name c_r_assessments
    $phase = Phase::where('is_active', 1)->first();

    $query = CRAssessment::with(['assessmentDetails.twentyTwoCR'])
        ->join('admissions', 'c_r_assessments.army_number', '=', 'admissions.army_number')
        ->select(
            'c_r_assessments.*',
            'admissions.pl',
            'admissions.name',
            'admissions.sec',
            'admissions.bn',
            'admissions.coy',
            'admissions.school_id',
            'admissions.course_id',
            'admissions.intake_id',
            'admissions.year_admission',
            'admissions.student_id'
        )
        ->whereNull('admissions.deleted_at')
        ->where('c_r_assessments.phase', $phase->name);

    // Apply filters if provided
    if ($platoon) {
        $query->where('admissions.pl', $platoon);
    }
    if ($coys) {
        $query->where('admissions.coy', $coys);
    }
    if ($section) {
        $query->where('admissions.sec', $section);
    }
    if ($bn) {
        $query->where('admissions.bn', $bn);
    }
    if ($schoolId) {
        $query->where('admissions.school_id', $schoolId);
    }
    if ($courseId) {
        $query->where('admissions.course_id', $courseId);
    }



    // Execute the query to get filtered and sorted assessments
    $assessments = $query->get()->groupBy('army_number'); // Grouping by 'army_number' for display

    // Generate the PDF and stream it to the browser for preview
    $pdf = Pdf::loadView('22cr.print', compact('assessments', 'platoon', 'coys', 'section', 'bn', 'schoolId', 'courseId', 'intakeId', 'yearAdmission', 'studentId'));

    return $pdf->stream('22cr.pdf'); // Opens PDF in browser (new tab)
}






}