<?php



namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Admission;
use App\Models\Assessment;
use App\Models\Course;
use App\Models\School;
use App\Models\Subject;
use App\Models\Exercise;
use App\Models\CRAssessment;
use Illuminate\Support\Facades\DB;
use App\Models\Pr;
use App\Models\TwentyTwoCR;
use Carbon\Carbon;
use App\Models\Phase;


class DashboardController extends Controller
{


public function index()
{

$admissionsCount = Admission::count();

    $warning = null;

    if ($admissionsCount === 0) {
        $warning = [
            'type' => 'error', // 'error' for critical, 'warning' for less severe
            'title' => 'CRITICAL: No Cadets Found in System!',
            'message' => 'STRART has zero cadet records. The dashboard and all assessment features require cadet data to function properly.',
            'link' => route('admissions.create'),
            'linkText' => 'Add Cadets Now',
            'importLink' => route('admissions.import'),
            'importText' => 'Bulk Import Cadets',
            'severity' => 'critical',
            'blocking_features' => true // Indicates this blocks main features
        ];
        
        // You might want to log this critical state
        \Log::warning('Dashboard accessed with zero cadets in system');
        
        // Optional: Flash a persistent message to session
        session()->flash('critical_error', 'No cadets found. Please add cadets to use the system.');

    }


    $phase  = Phase::where('is_active', 1)->get()->first();
$intake = \App\Models\Intake::where('is_active', 1)->first(); // ADD THIS

    // Basic counts
    $totalCadets = Admission::count();
    $totalSchools = School::count();
    $totalsubjects = Subject::count();
   $exercises = Exercise::count();
    // Get best performing groups with recent assessments (last 30 days)
    $bestGroups = [
        'battalions' => Assessment::select(
                'bn',
                DB::raw('COUNT(DISTINCT admission_id) as total_cadets'),
                DB::raw('AVG(marks) as average_marks')
            )
            
            ->whereDate('created_at', '>=', now()->subDays(30))
            ->groupBy('bn')
            ->orderBy('average_marks', 'desc')
            ->get(),

        'companies' => Assessment::select(
                'coy',
                DB::raw('COUNT(DISTINCT admission_id) as total_cadets'),
                DB::raw('AVG(marks) as average_marks')
            )
            ->whereDate('created_at', '>=', now()->subDays(30))
            ->groupBy('coy')
            ->orderBy('average_marks', 'desc')
            ->get(),

        'platoons' => Assessment::select(
                'pl',
                DB::raw('COUNT(DISTINCT admission_id) as total_cadets'),
                DB::raw('AVG(marks) as average_marks')
            )
            ->whereDate('created_at', '>=', now()->subDays(30))
            ->groupBy('pl')
            ->orderBy('average_marks', 'desc')
            ->get(),

        'sections' => Assessment::select(
                'sec',
                DB::raw('COUNT(DISTINCT admission_id) as total_cadets'),
                DB::raw('AVG(marks) as average_marks')
            )
            ->whereDate('created_at', '>=', now()->subDays(30))
            ->groupBy('sec')
            ->orderBy('average_marks', 'desc')
            ->get()
    ];


    $topPerformers = Assessment::select(
        'admissions.name',
        'admissions.army_number',
        'assessments.bn',
        'assessments.coy',
        'assessments.pl',
        'assessments.sec',
        'assessments.subject_id',
        'assessments.type',
        'assessments.week',
        'assessments.marks',
        'subjects.name as subject_name'
    )
    ->join('admissions', 'assessments.admission_id', '=', 'admissions.id')
    ->join('subjects', 'assessments.subject_id', '=', 'subjects.id')
    ->whereNotNull('marks')
    ->whereNotNull('subject_id')
    ->orderBy('marks', 'desc')
    ->take(5)
    ->get()
    ->groupBy('subject_id');

    // Get worst performers by subject
    $worstPerformers = Assessment::select(
        'admissions.name',
        'admissions.army_number',
        'assessments.bn',
        'assessments.coy',
        'assessments.pl',
        'assessments.sec',
        'assessments.subject_id',
        'assessments.type',
        'assessments.week',
        'assessments.marks',
        'subjects.name as subject_name'
    )
    ->join('admissions', 'assessments.admission_id', '=', 'admissions.id')
    ->join('subjects', 'assessments.subject_id', '=', 'subjects.id')
    ->whereNotNull('marks')
    ->whereNotNull('subject_id')
    ->orderBy('marks', 'asc')
    ->take(5)
    ->get()
    ->groupBy('subject_id');

  $totalAssessments = Assessment::join('admissions', 'assessments.admission_id', '=', 'admissions.id')
    ->whereNull('admissions.deleted_at')
    ->count();

    $passedAssessments = Assessment::join('admissions', 'assessments.admission_id', '=', 'admissions.id')
    ->whereNull('admissions.deleted_at')
    ->where('marks', '>=', 60)
    ->count();

    $metrics = [
        'passRate' => $totalAssessments > 0
            ? (Assessment::where('marks', '>=', 60)->count() * 100 / $totalAssessments)
            : 0,
'averageScore' => Assessment::join('admissions', 'assessments.admission_id', '=', 'admissions.id')
    ->whereNull('admissions.deleted_at')
    ->avg('marks') ?? 0,

   'todayAssessments' => Assessment::join('admissions', 'assessments.admission_id', '=', 'admissions.id')
    ->whereNull('admissions.deleted_at')
    ->whereDate('assessments.created_at', today())
    ->count(),

        'improvementRate' => $this->calculateImprovementRate()
    ];

    // Recent assessments
 $recentAssessments = Assessment::with(['admission', 'subject'])
    ->join('admissions', 'assessments.admission_id', '=', 'admissions.id')
    ->whereNull('admissions.deleted_at')
    ->latest('assessments.created_at')
    ->take(5)
    ->get();


    // Chart data
    $chartData = [
        'labels' => ['Week 1', 'Week 2', 'Week 3', 'Week 4','Week 5','Week 6','Week 7','Week 8','Week 9','Week 10'],
        'performance' => Assessment::select(DB::raw('AVG(marks) as avg_marks'))
            ->whereDate('created_at', '>=', now()->subWeeks(4))
            ->groupBy(DB::raw('WEEK(created_at)'))
            ->pluck('avg_marks')
            ->toArray(),
        'distribution' => [
            Assessment::where('marks', '>=', 80)->count(),
            Assessment::whereBetween('marks', [70, 79.99])->count(),
            Assessment::whereBetween('marks', [60, 69.99])->count(),
            Assessment::where('marks', '<', 60)->count(),
        ]
    ];

      // Get top performers by assessment type
      $topPerformersByType = [
        'Weekly' => $this->getTopPerformers('Weekly'),
        'Monthly' => $this->getTopPerformers('Monthly'),
        'End of Phase Exam' => $this->getTopPerformers('End of Phase Exam')
    ];

    // Get top performers by subject and assessment group
    $topPerformersBySubject = $this->getTopPerformersBySubject();

    // Get top performers by assessment group
    $topPerformersByGroup = [
        'Theory' => $this->getTopPerformersByAssessmentGroup('Theory'),
        'Practical' => $this->getTopPerformersByAssessmentGroup('Practical')
    ];

    // Get top performers by type (removed CR reference)
    $topPerformersByType = [
        'Weekly' => $this->getTopPerformersByType('Weekly'),
        'Monthly' => $this->getTopPerformersByType('Monthly'),
        'End of Phase Exam' => $this->getTopPerformersByType('End of Phase Exam'),

    ];



    $topPerformersByTypes = [
        '22CR' => $this->getTopPerformersBy22CR(),
        'PR' => $this->getTopPerformersByPR(),
    ];

    $topPerformersByExercise = $this->getTopPerformersByExercise();

    $bestOfBest = $this->getBestOfBestPerformers();

    $bestPlatoons = $this->getBestPlatoons();


//    $top22CRPerformers = $this->getTopPerformersBy22CR();
//    $topPRPerformers = $this->getTopPerformersByPR();
//
//    return dd( $topPRPerformers);

    return view('dashbaord', compact(
      'warning',
        'bestPlatoons',
        'totalCadets',
        'bestOfBest',
        'topPerformersByExercise',
        'topPerformersByType',
        'topPerformersByTypes',
        'totalSchools',
        'totalsubjects',
        'topPerformersBySubject',
        'exercises',
        'bestGroups',
        'topPerformers',
        'metrics',
        'recentAssessments',
        'chartData'
    ));
}

private function getTopPerformersByExercise($limit = 5)
{
    $exerciseTypes = ['PT', 'SAA', 'MU', 'TAC', 'DR'];
    $results = [];

    $phase  = Phase::where('is_active', 1)->get()->first();
    foreach ($exerciseTypes as $exercise) {
        $performers = Assessment::select(
            'assessments.id',
            'assessments.admission_id',
            'assessments.marks',
            'assessments.type',
            'assessments.week',
            'assessments.assessment_group',
            'assessments.bn',
            'assessments.coy',
            'assessments.pl',
            'assessments.sec',
            'assessments.exercises',
            'admissions.name',
            'admissions.army_number'
        )
        ->join('admissions', 'assessments.admission_id', '=', 'admissions.id')
        ->where('exercises', $exercise)
        ->whereNull('admissions.deleted_at')
        ->whereNotNull('marks')
        ->orderByDesc('marks')
        ->limit($limit)
        ->get();

        $results[$exercise] = [
            'exercise_name' => $exercise,
            'performers' => $performers,
            'average' => $performers->avg('marks'),
            'pass_rate' => $performers->where('marks', '>=', 60)->count() / max(1, $performers->count()) * 100
        ];
    }

    return $results;
}

    private function calculateImprovementRate(?Carbon $referenceDate = null): float
    {
        // Use provided date or default to current date
        $referenceDate = $referenceDate ?? now();

        try {
            // Validate and retrieve previous month's average marks
            $previousMonth = $this->getMonthlyAverage($referenceDate->subMonth());

            // Validate and retrieve current month's average marks
            $currentMonth = $this->getMonthlyAverage($referenceDate);

            // Handle edge cases with detailed logic
            if ($previousMonth === 0.0) {
                // If no previous assessments, different handling based on current month's performance
                if ($currentMonth > 0) {
                    return 100.0; // Significant improvement from zero baseline
                }
                return 0.0; // No change if both months are zero
            }

            // Calculate improvement rate with decimal precision
            $improvementRate = (($currentMonth - $previousMonth) / $previousMonth) * 100;

            // Round to two decimal places for readability
            return round($improvementRate, 2);

        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('Improvement Rate Calculation Error: ' . $e->getMessage());

            // Return a neutral value or throw a custom exception
            return 0.0;
        }
    }

    /**
     * Retrieve monthly average marks with additional error handling
     *
     * @param Carbon $date
     * @return float
     */
    private function getMonthlyAverage(Carbon $date): float
    {
        try {
            $average = Assessment::whereMonth('created_at', '=', $date->month)
                ->whereYear('created_at', '=', $date->year)
                ->avg('marks');

            // Ensure float type and handle potential null
            return (float) ($average ?? 0.0);

        } catch (\Exception $e) {
            // Log specific query errors
            \Log::warning('Monthly Average Calculation Failed: ' . $e->getMessage());
            return 0.0;
        }
    }

private function getTopPerformers($type, $limit = 5)
{
    return Assessment::select(
        'admissions.name',
        'admissions.army_number',
        'assessments.bn',
        'assessments.coy',
        'assessments.pl',
        'assessments.sec',
        'assessments.type',
        'assessments.week',
        'assessments.marks',
        'assessments.assessment_group',
        'subjects.name as subject_name',
        'exercises'
    )
    ->join('admissions', 'assessments.admission_id', '=', 'admissions.id')
    ->leftJoin('subjects', 'assessments.subject_id', '=', 'subjects.id')
    ->whereNull('admissions.deleted_at')
    ->where('type', $type)
    ->orderBy('marks', 'desc')
    ->take($limit)
    ->get();
}

private function getTopPerformersBySubject($limit = 5)
{
    return Subject::with(['assessments' => function($query) use ($limit) {
        $query->select(
            'assessments.*',
            'admissions.name',
            'admissions.army_number'
        )
        ->join('admissions', 'assessments.admission_id', '=', 'admissions.id')
        ->whereNull('admissions.deleted_at')
        ->orderBy('marks', 'desc')
        ->take($limit);
    }])->get()->mapWithKeys(function($subject) {
        return [$subject->name => $subject->assessments];
    });
}



private function getTopPerformersByAssessmentGroup($group, $limit = 5)
{
    return Assessment::select(
        'admissions.name',
        'admissions.army_number',
        'assessments.bn',
        'assessments.coy',
        'assessments.pl',
        'assessments.sec',
        'assessments.type',
        'assessments.week',
        'assessments.marks',
        'subjects.name as subject_name',
        'exercises'
    )
    ->join('admissions', 'assessments.admission_id', '=', 'admissions.id')
    ->leftJoin('subjects', 'assessments.subject_id', '=', 'subjects.id')
    ->whereNull('admissions.deleted_at')
    ->where('assessment_group', $group)
    ->orderBy('marks', 'desc')
    ->take($limit)
    ->get();
}
    private function getTopPerformersBy22CR($limit = 5)
    {
        $students = Admission::all(); // Get all students
        $maxScore = 20; // Adjust based on your scoring system

        $performers = $students->map(function ($student) use ($maxScore) {
            return [
                'name' => $student->name,
                'army_number' => $student->army_number,
                'score' => $this->calculate22CRScore($student, $maxScore),
            ];
        });

        // Sort by score in descending order and get the top 5
        return collect($performers)->sortByDesc('score')->take($limit)->values();
    }

private function calculate22CRScore($student, $maxScore)
{
    $assessments = CRAssessment::where('army_number', $student->army_number)
        ->with(['assessmentDetails.twentyTwoCR'])
        ->get();

    if ($assessments->isEmpty()) {
        return 0;
    }

    $totalCRES = 0;

    foreach ($assessments as $assessment) {
        foreach ($assessment->assessmentDetails as $detail) {
            // Check if twentyTwoCR relationship exists and has weight
            if (!$detail->twentyTwoCR || !isset($detail->twentyTwoCR->weight)) {
                continue; // Skip this detail if no twentyTwoCR or weight
            }

            $f = $detail->f1 + $detail->f2 + $detail->f3 + $detail->f4 + $detail->f5;
            $fx = ($detail->f1 * 1) + ($detail->f2 * 2) + ($detail->f3 * 3) + ($detail->f4 * 4) + ($detail->f5 * 5);

            if ($f > 0) {
                $individualCRES = ($fx / $f) * $detail->twentyTwoCR->weight;
                $totalCRES += $individualCRES;
            }
        }
    }

    $totalPossibleCRES = 720; // Assuming 720 is the maximum CRES
    return round(($totalCRES / $totalPossibleCRES) * $maxScore, 2);
}
    private function getTopPerformersByPR($limit = 5)
    {
        $students = Admission::all(); // Get all students
        $maxScore = 10; // Adjust based on your scoring system

        $performers = $students->map(function ($student) use ($maxScore) {
            return [
                'name' => $student->name,
                'army_number' => $student->army_number,
                'score' => $this->calculatePrScore($student, $maxScore),
            ];
        });

        // Sort by score in descending order and get the top 5
        return collect($performers)->sortByDesc('score')->take($limit)->values();
    }
    private function calculatePrScore($student, $maxScore)
    {
        // Get PR ratings for this student
        $prRatings = PR::where('army_number', $student->army_number)->get();

        if ($prRatings->isEmpty()) {
            return 0;
        }

        $totalScore = $prRatings->sum('score');
        $totalPeers = Admission::where('bn', $student->bn)
            ->where('coy', $student->coy)
            ->where('pl', $student->pl)
            ->where('id', '!=', $student->id)
            ->count();

        // Calculate mean rating
        $meanRating = $totalPeers > 0 ? $totalScore / $totalPeers : 0;

        if ($meanRating <= 0) {
            return 0;
        }

        // Weight table for PR scoring
        $assignedWeights = [
            1 => 6,
            2 => 10,
            3 => 14,
            4 => 18,
            5 => 22,
            6 => 26,
            7 => 30
        ];

        $wholeNumber = floor($meanRating);
        $decimalPart = $meanRating - $wholeNumber;

        if (!isset($assignedWeights[$wholeNumber])) {
            return 0;
        }

        $baseWeight = $assignedWeights[$wholeNumber];
        $additionalWeight = $decimalPart * 4; // Constant interval is 4
        $totalWeight = $baseWeight + $additionalWeight;

        // Scale to max_score (10% weight)
        return round(($totalWeight / 30) * $maxScore, 1);
    }


private function getTopPerformersByType($type, $limit = 5)
{
    return Assessment::select(
        'assessments.*',
        'admissions.name',
        'admissions.army_number',
        'subjects.name as subject_name'
    )
    ->join('admissions', 'assessments.admission_id', '=', 'admissions.id')
    ->leftJoin('subjects', 'assessments.subject_id', '=', 'subjects.id')
    ->whereNull('admissions.deleted_at')
    ->where('type', $type)
    ->orderByDesc('marks')
    ->take($limit)
    ->get();
}

private function getBestOfBestPerformers($limit = 5)
{
    $performanceController = app(StudentPerformanceController::class);
    
    $phase = Phase::where('is_active', 1)->first();
    
    if (!$phase) {
        return collect();
    }

    $students = Admission::where(function($query) {
            $query->where('is_decommissioned', 0)
                  ->orWhereNull('is_decommissioned');
        })
        ->whereNull('deleted_at')
        ->get();

    $performers = [];

    foreach ($students as $student) {
        $scores = $performanceController->calculateStudentOverallScore($student, $phase->name);
        
        if ($scores && $scores['total'] > 0) {
            $performers[] = (object)[
                'id' => $student->id,
                'admission_id' => $student->id,
                'marks' => $scores['total'],
                'type' => 'Overall Performance',
                'week' => 'All',
                'assessment_group' => 'Composite',
                'exercises' => 'GPV Score',
                'bn' => $student->bn,
                'coy' => $student->coy,
                'pl' => $student->pl,
                'name' => $student->name,
                'army_number' => $student->army_number,
                'subject_name' => 'Overall Grade',
                'performance_type' => 'composite',
                'theory_score' => $scores['theory'],
                'practical_score' => $scores['practical'],
                'cr22_score' => $scores['cr22'],
                'pr_score' => $scores['pr'],
                'total_score' => $scores['total'],
                'grade' => $scores['grade'],
                'comment' => $scores['comment']
            ];
        }
    }

    usort($performers, function($a, $b) {
        return $b->marks <=> $a->marks;
    });

    return collect(array_slice($performers, 0, $limit));
}

private function getBestPlatoons()
{
    $performanceController = app(StudentPerformanceController::class);
    $phase = Phase::where('is_active', 1)->first();
    
    if (!$phase) {
        return collect();
    }

    // Get all active students with their platoon information
    $students = Admission::where(function($query) {
            $query->where('is_decommissioned', 0)
                  ->orWhereNull('is_decommissioned');
        })
        ->whereNotNull('pl')
        ->whereNull('deleted_at')
        ->get();

    // Group students by platoon and calculate overall scores
    $platoonData = [];

    foreach ($students as $student) {
        if (empty($student->pl)) {
            continue;
        }

        $platoon = $student->pl;

        // Calculate overall score using the same GPV logic
        $maxScores = $performanceController->getMaxScores();
        $theoryScore = $performanceController->calculateTheoryScore($student, $maxScores['theory'], $phase->name);
        $practicalScore = $performanceController->calculatePracticalScore($student->id, $phase->name);
        $cr22Score = $performanceController->calculate22CRScore($student, $maxScores['22cr'], $phase->name);
        $prScore = $performanceController->calculatePrScore($student, $maxScores['pr'], $phase->name);

        $totalScore = $theoryScore + $practicalScore + $cr22Score + $prScore;

        // Initialize platoon data if not exists
        if (!isset($platoonData[$platoon])) {
            $platoonData[$platoon] = [
                'pl' => $platoon,
                'total_cadets' => 0,
                'total_score_sum' => 0,
                'theory_score_sum' => 0,
                'practical_score_sum' => 0,
                'cr22_score_sum' => 0,
                'pr_score_sum' => 0,
                'cadet_scores' => []
            ];
        }

        // Add student data to platoon
        $platoonData[$platoon]['total_cadets']++;
        $platoonData[$platoon]['total_score_sum'] += $totalScore;
        $platoonData[$platoon]['theory_score_sum'] += $theoryScore;
        $platoonData[$platoon]['practical_score_sum'] += $practicalScore;
        $platoonData[$platoon]['cr22_score_sum'] += $cr22Score;
        $platoonData[$platoon]['pr_score_sum'] += $prScore;
        $platoonData[$platoon]['cadet_scores'][] = $totalScore;
    }

    // Calculate averages and prepare final data
    $allPlatoons = collect($platoonData)->map(function ($platoon) {
        $totalCadets = $platoon['total_cadets'];
        
        return [
            'pl' => $platoon['pl'],
            'total_cadets' => $totalCadets,
            'average_marks' => $totalCadets > 0 ? $platoon['total_score_sum'] / $totalCadets : 0,
            'component_averages' => [
                'theory' => $totalCadets > 0 ? $platoon['theory_score_sum'] / $totalCadets : 0,
                'practical' => $totalCadets > 0 ? $platoon['practical_score_sum'] / $totalCadets : 0,
                '22cr' => $totalCadets > 0 ? $platoon['cr22_score_sum'] / $totalCadets : 0,
                'pr' => $totalCadets > 0 ? $platoon['pr_score_sum'] / $totalCadets : 0,
            ],
            'performance_by_type' => [
                'Overall GPV' => [
                    'average' => $totalCadets > 0 ? $platoon['total_score_sum'] / $totalCadets : 0,
                    'cadets' => $totalCadets
                ],
                'Theory' => [
                    'average' => $totalCadets > 0 ? $platoon['theory_score_sum'] / $totalCadets : 0,
                    'cadets' => $totalCadets
                ],
                'Practical' => [
                    'average' => $totalCadets > 0 ? $platoon['practical_score_sum'] / $totalCadets : 0,
                    'cadets' => $totalCadets
                ],
                '22CR' => [
                    'average' => $totalCadets > 0 ? $platoon['cr22_score_sum'] / $totalCadets : 0,
                    'cadets' => $totalCadets
                ],
                'PR' => [
                    'average' => $totalCadets > 0 ? $platoon['pr_score_sum'] / $totalCadets : 0,
                    'cadets' => $totalCadets
                ]
            ],
            'score_distribution' => [
                'min' => count($platoon['cadet_scores']) > 0 ? min($platoon['cadet_scores']) : 0,
                'max' => count($platoon['cadet_scores']) > 0 ? max($platoon['cadet_scores']) : 0,
                'median' => $this->calculateMedian($platoon['cadet_scores'])
            ]
        ];
    })->sortByDesc('average_marks')->values();

    return $allPlatoons;
}

// Helper method to calculate median
private function calculateMedian($scores)
{
    if (empty($scores)) {
        return 0;
    }

    sort($scores);
    $count = count($scores);
    $middle = floor($count / 2);

    if ($count % 2 == 0) {
        // Even number of elements
        return ($scores[$middle - 1] + $scores[$middle]) / 2;
    } else {
        // Odd number of elements
        return $scores[$middle];
    }
}
}
