<?php

namespace App\Http\Controllers;

use App\Models\Assessment;
use App\Models\Admission;
use App\Exports\AssessmentsExport;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\PDF;
use Maatwebsite\Excel\Facades\Excel;

class ExportController extends Controller
{
    /**
     * Export assessments data based on filters to Excel
     */
    public function exportExcel(Request $request)
    {
        $fileName = 'assessments_' . date('Y-m-d_His') . '.xlsx';
        return Excel::download(new AssessmentsExport($request->all()), $fileName);
    }

    /**
     * Export assessments data based on filters to PDF
     */
    public function exportPDF(Request $request)
    {
        // Get filtered data
        $assessments = $this->getFilteredAssessments($request);

        // Determine columns based on assessment type
        $type = $request->input('type', '');
        $columns = $this->getColumnsForType($type, $assessments);

        // Create PDF
        $pdf = PDF::loadView('exports.assessments_pdf', [
            'assessments' => $assessments,
            'type' => $type,
            'columns' => $columns,
            'filters' => $request->all()
        ]);

        return $pdf->download('assessments_' . date('Y-m-d_His') . '.pdf');
    }

    /**
     * Get columns based on assessment type
     */
    private function getColumnsForType($type, $assessments)
    {
        if ($type == 'Monthly') {
            // Get unique monthly columns
            return $assessments->first()?->filter(function($assessment) {
                return $assessment->type === 'Monthly';
            })->pluck('month')->unique()->sort() ?? collect([]);
        } elseif ($type == 'End of Phase Exam') {
            return ['EOPE'];
        } else {
            // Weekly columns
            return collect(range(1, 10))->map(function($i) {
                return 'Week ' . $i;
            });
        }
    }

    /**
     * Get filtered assessments based on request
     */
    private function getFilteredAssessments(Request $request)
    {
        $query = Assessment::with('admission');

        // Apply filters
        if ($request->filled('assessment_type')) {
            $query->where('assessment_type', $request->input('assessment_type'));
        }

        if ($request->filled('assessment_group')) {
            $query->where('assessment_group', $request->input('assessment_group'));
        }

        if ($request->filled('type')) {
            $query->where('type', $request->input('type'));
        }

        if ($request->filled('week')) {
            $query->where('week', $request->input('week'));
        }

        if ($request->filled('subject_id')) {
            $query->where('subject_id', $request->input('subject_id'));
        }

        if ($request->filled('exercises')) {
            $query->where('exercise', $request->input('exercises'));
        }

        // Organization filters
        if ($request->filled('bn')) {
            $query->where('bn', $request->input('bn'));
        }

        if ($request->filled('coy')) {
            $query->where('coy', $request->input('coy'));
        }

        if ($request->filled('pl')) {
            $query->where('pl', $request->input('pl'));
        }

        if ($request->filled('sec')) {
            $query->where('sec', $request->input('sec'));
        }

        // Get results grouped by student
        return $query->get()->groupBy('admission_id');
    }
}
