<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Intake;
use App\Models\Phase;
use App\Models\Admission;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Models\GpvReport;
use App\Models\GpvReportDetail;

class IntakeController extends Controller
{
    // Show all intakes
    public function index()
    {
        $intakes = Intake::all();
        return view('schools.index', compact('intakes'));
    }

    // Show form to create new intake
    public function create()
    {
        return view('intakes.create');
    }

/**
 * Activate an existing intake and archive the previously active intake's data.
 *
 * @param  \Illuminate\Http\Request  $request
 * @return \Illuminate\Http\Response
 */
/**
 * Activate an existing intake and archive the previously active intake's data.
 *
 * @param  \Illuminate\Http\Request  $request
 * @return \Illuminate\Http\Response
 */
public function store(Request $request)
{
    // Validate that the year is in correct format and doesn't already exist
    $validated = $request->validate([
        'year' => 'required|string|regex:/^\d{2}\/\d{2}$/|unique:intakes,year',
    ]);

    DB::beginTransaction();
    try {
        // Get the currently active intake (if any)
        $currentlyActive = Intake::where('is_active', 1)->first();

        // --- Check if we can create a new intake ---
        if ($currentlyActive) {
            $activePhase = Phase::where('intake_id', $currentlyActive->id)
                                ->where('is_active', 1)
                                ->first();
            
            // Only allow creating new intake if Phase 3 is active
            if (!$activePhase || $activePhase->name !== 'Phase 3') {
                DB::rollBack();
                return redirect()->back()
                       ->with('warning', 'Cannot create new intake yet. Phase 3 must be active in the current intake.');
            }
            
            // Get all admissions from the active intake
            $admissions = Admission::where('intake_id', $currentlyActive->id)->get();
            
            foreach ($admissions as $admission) {
                // 1. Delete from pivot tables first (many-to-many relationships)
                DB::table('admission_subject')
                    ->where('admission_id', $admission->id)
                    ->delete();
                
                DB::table('admission_phases')
                    ->where('admission_id', $admission->id)
                    ->delete();
                
                // 2. Delete related records from child tables
                DB::table('assessments')
                    ->where('admission_id', $admission->id)
                    ->delete();
                
                DB::table('prs')
                    ->where('admission_id', $admission->id)
                    ->delete();
                
                DB::table('c_r_assessment2s')
                    ->where('admission_id', $admission->id)
                    ->delete();
                
    
                // 4. Handle the associated student (if no other admissions reference this student)
                $otherAdmissionsCount = Admission::where('student_id', $admission->student_id)
                    ->where('id', '!=', $admission->id)
                    ->count();
                
                if ($otherAdmissionsCount === 0 && $admission->student_id) {
                    // Delete student if no other admissions reference them
                    DB::table('students')
                        ->where('id', $admission->student_id)
                        ->delete();
                }
                
                // 5. Finally delete the admission itself
                $admission->forceDelete(); // Use forceDelete to ensure hard delete even if using SoftDeletes
            }
            
            // Optional: Log the cleanup action
            \Log::info("Permanently deleted " . count($admissions) . " admissions and related records from intake ID: {$currentlyActive->id}");
        }

        // --- Create the new intake ---
        $newIntake = Intake::create([
            'year' => $request->year,
            'is_active' => 0, // Start as inactive
        ]);

        // Get the latest intake to copy phases from (if any)
        $latestIntake = Intake::where('id', '!=', $newIntake->id)
                          ->latest('id')
                          ->first();

        if ($latestIntake) {
            // Copy all phases from the latest intake to the new intake
            $phases = Phase::where('intake_id', $latestIntake->id)->get();
            
            foreach ($phases as $phase) {
                Phase::create([
                    'name' => $phase->name,
                    'weeks' => $phase->weeks,
                    'cres_weight' => $phase->cres_weight,
                    'pr_weight' => $phase->pr_weight,
                    'intake_id' => $newIntake->id,
                    'is_active' => 0, // Start with all phases inactive
                ]);
            }
        }

        DB::commit();

        return redirect()->route('schools.index')
               ->with('success', "New intake {$newIntake->year} created successfully. Previous intake data has been permanently deleted.");
               
    } catch (\Exception $e) {
        DB::rollBack();
        return redirect()->back()->with('error', 'An error occurred: ' . $e->getMessage());
    }
}

    /**
     * Force create a new intake even if not on last phase
     * 
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function forceStore(Request $request)
    {
        // Validate the request
        $validated = $request->validate([
            'year' => 'required|integer|min:2000|max:2100|unique:intakes,year',
        ]);

        DB::beginTransaction();
        try {
            // Create the new intake
            $newIntake = Intake::create([
                'year' => $request->year
            ]);

            // Get the latest intake to copy phases from (if any)
            $latestIntake = Intake::where('id', '!=', $newIntake->id)
                              ->latest('id')
                              ->first();

            if ($latestIntake) {
                // Copy all phases from the latest intake to the new intake
                $phases = Phase::where('intake_id', $latestIntake->id)->get();
                
                foreach ($phases as $phase) {
                    Phase::create([
                        'name' => $phase->name,
                        'weeks' => $phase->weeks,
                        'cres_weight' => $phase->cres_weight,
                        'pr_weight' => $phase->pr_weight,
                        'intake_id' => $newIntake->id,
                        'is_active' => 0, // Start with all phases inactive
                        'current' => 0,
                    ]);
                }
                
                // Start the first phase of the new intake
                $firstPhase = Phase::where('intake_id', $newIntake->id)
                                 ->orderBy('id', 'asc')
                                 ->first();
                
                if ($firstPhase) {
                    $firstPhase->update(['is_active' => 1]);
                    $newIntake->update(['current' => $firstPhase->id]);
                }
            }
            
            DB::commit();
            return redirect()->route('schools.index')
                   ->with('success', 'New intake created successfully.');
                   
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                   ->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }
    // Show form to edit intake
    public function edit(Intake $intake)
    {
        return view('intakes.edit', compact('intake'));
    }

    // Update intake
    public function update(Request $request, Intake $intake)
    {
     

        $intake->update(['year' => $request->year]);

        return redirect()->route('schools.index')->with('success', 'Intake updated successfully.');
    }


    public function getIntakes(Request $request) {
        return Intake::pluck('year', 'id');
    }

    // Delete intake
    public function destroy(Intake $intake)
    {
        $intake->delete();
        return redirect()->route('schools.index')->with('success', 'Intake deleted successfully.');
    }
}
