<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Assessment;
use App\Models\Admission;
use App\Models\School;
use App\Models\Course;
use App\Models\Intake;
use App\Models\Subject;
use App\Models\Bn;
use App\Models\Coy;
use App\Models\Sec;
use App\Models\Pl;
use App\Models\TwentyTwoCR;
use App\Models\ExamWeight;
use App\Models\Exercise;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use App\Models\Phase;

use Illuminate\Validation\ValidationException;
// exam_weights exercises
use Illuminate\Support\Facades\DB;



class MarkEntryController extends Controller
{


    public function create2(Request $request)
    {
        // Get the selected phase from the request (if present)
        $selectedPhase = $request->get('phase');

        // Fetch the necessary data for dropdowns
        $schools = School::pluck('name', 'id');
        $courses = Course::pluck('name', 'id');
        $intakes = Intake::pluck('year', 'id');
        $subjects = Subject::select('name', 'id')->get();
        $bn = Bn::pluck('name');
        $coy = Coy::pluck('name');
        $sec = Sec::pluck('name');
        $pl = Pl::pluck('name');
        $twentyTwoCR = $selectedPhase ? TwentyTwoCR::where('phase', $selectedPhase)->get() : collect();

        // Get all phases for the dropdown
        $phases = Phase::all();

        // Create an associative array of phases and their respective weeks
        $phaseWeeks = [];
        foreach ($phases as $phase) {
            $phaseWeeks[$phase->name] = range(1, $phase->weeks);
        }

        // Get exercise titles
        $exercise = Exercise::pluck('title');

        // Return the view with the selected phase and other data
        return view('marks_entry', compact(
            'phaseWeeks',
            'schools',
            'courses',
            'intakes',
            'phases',
            'exercise',
            'subjects',
            'twentyTwoCR',
            'bn',
            'coy',
            'sec',
            'pl',
            'selectedPhase' // Make sure this is passed to the view
        ));
    }


public function create(Request $request)
{
    // Fetch dropdown data
    $schools = School::pluck('name', 'id');
    $courses = Course::pluck('name', 'id');
    $intakes = Intake::pluck('year', 'id');
    $subjects = Subject::select('name', 'id')->get();

    $bn = Bn::pluck('name');
    $coy = Coy::pluck('name');
    $sec = Sec::pluck('name');
    $pl = Pl::pluck('name');

    // Get first active phase (can be null)
    $phase  = Phase::where('is_active', 1)->first();
    $phases = Phase::all(); // For dropdown

    $selectedPhase = $request->get('phase');

    // Only query exercises if $phase exists
    if ($phase) {
        $exercise = Exercise::where('phase_id', $phase->id)->pluck('title');
    } else {
        $exercise = collect(); // empty collection to avoid errors
    }

    $twentyTwoCR = TwentyTwoCR::select('id', 'title', 'weight')->get();

    return view('marks_entry2', compact(
        'schools',
        'courses',
        'phases',
        'intakes',
        'phase',
        'exercise',
        'subjects',
        'twentyTwoCR',
        'selectedPhase',
        'bn',
        'coy',
        'sec',
        'pl',
    ));
}
public function store(Request $request)
{
    // Validate the request
    $request->validate([
        'school_id' => 'required|exists:schools,id',
        'course_id' => 'required|exists:courses,id',
        'intake_id' => 'required|exists:intakes,id',
        'assessment_type' => 'required|string',
        'assessment_group' => 'required|string',
        'type' => 'required|string',
        'subject_id' => 'required|exists:subjects,id',
        'week' => 'nullable|string',
        'exercises' => 'nullable|string',
        'month' => 'nullable|string|in:Jan,Feb,Mar,Apr,May,Jun,Jul,Aug,Sep,Oct,Nov,Dec',
        'students' => 'required|array',
        'students.*.student_id' => 'required|exists:admissions,id',
        'students.*.marks' => 'nullable|integer|min:0|max:100',
    ]);

    DB::beginTransaction();

    $phase = Phase::where('is_active', 1)->first();

    try {
        $subjectId = $request->subject_id;
        $assessmentGroup = $request->assessment_group;
        $assessmentType = $request->assessment_type;

        // Track counts for feedback
        $newCount = 0;
        $updatedCount = 0;
        $skippedCount = 0;
        $processedIds = [];

        // Process each student's marks
        foreach ($request->students as $studentData) {
            $studentId = $studentData['student_id'];
            $marks = $studentData['marks'] ?? null;
            
            // Skip duplicates in same request
            if (in_array($studentId, $processedIds)) {
                $skippedCount++;
                continue;
            }
            $processedIds[] = $studentId;
            
            // Prepare the unique data for this assessment
            $uniqueData = [
                'admission_id' => $studentId,
                'subject_id' => $subjectId,
                'assessment_group' => $assessmentGroup,
                'phase' => $phase->name,
                'type' => $request->type,
                'assessment_type' => $assessmentType,
            ];
            
            // Add optional fields based on type
            if ($request->type === 'Weekly' && $request->filled('week')) {
                $uniqueData['week'] = $request->week;
            } elseif ($request->type === 'Monthly' && $request->filled('month')) {
                $uniqueData['month'] = $request->month;
            }
            
            if ($request->filled('exercises')) {
                $uniqueData['exercises'] = $request->exercises;
            }
            
            // Use the new updateOrCreateUnique method from our model
            $assessment = Assessment::updateOrCreateUnique(
                $uniqueData,  // Unique conditions
                ['marks' => $marks]  // Data to update/create
            );
            
            if ($assessment->wasRecentlyCreated) {
                $newCount++;
            } else {
                $updatedCount++;
            }
        }

        DB::commit();
        
        $message = 'Marks entered successfully!';
        $parts = [];
        if ($newCount > 0) $parts[] = "$newCount new";
        if ($updatedCount > 0) $parts[] = "$updatedCount updated";
        if ($skippedCount > 0) $parts[] = "$skippedCount skipped (duplicates in request)";
        
        if (!empty($parts)) {
            $totalProcessed = count($processedIds);
            $message .= " (" . implode(', ', $parts) . " out of $totalProcessed students)";
        }
        
        return redirect()->route('marks_entry.create')->with('success', $message);
        
    } catch (\Illuminate\Validation\ValidationException $e) {
        DB::rollBack();
        return back()->withErrors($e->getMessage());
    } catch (\Illuminate\Database\QueryException $e) {
        DB::rollBack();
        
        if ($e->getCode() == 23000) { // Duplicate entry error
            return back()->withErrors('Duplicate assessment detected. These marks may already exist in the system.');
        }
        
        return back()->withErrors('Database error: ' . $e->getMessage());
    } catch (\Exception $e) {
        DB::rollBack();
        return back()->withErrors('Error while saving marks: ' . $e->getMessage());
    }
}






public function saveMarks(Request $request)
{
    try {
        // Validation rules
        $rules = [
            'marks' => 'required|array|min:1',
            'marks.*.admission_id' => 'required|integer|exists:admissions,id',
            'marks.*.marks' => 'required|numeric|min:0|max:100',
            'phase' => 'required|string|max:100',
            'assessment_type' => 'required|string|max:100',
            'assessment_group' => 'required|string|max:100',
            'type' => 'required|string|in:Monthly,Weekly,End of Phase Exam',
            'subject_id' => 'nullable|integer|exists:subjects,id',
            'code_name' => 'nullable|string|max:100',
            'exercises' => 'nullable|string|max:255',
            'week' => 'nullable|string|max:50',
            'month' => 'nullable|string|in:Jan,Feb,Mar,Apr,May,Jun,Jul,Aug,Sep,Oct,Nov,Dec',
            'bn' => 'nullable|string|max:50',
            'coy' => 'nullable|string|max:50',
            'pl' => 'nullable|string|max:50',
            'sec' => 'nullable|string|max:50',
        ];

        // Conditional validation
        if ($request->input('type') === 'Monthly') {
            $rules['month'] = 'required|string|in:Jan,Feb,Mar,Apr,May,Jun,Jul,Aug,Sep,Oct,Nov,Dec';
        } elseif ($request->input('type') === 'Weekly') {
            $rules['week'] = 'required|string|max:50';
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => $validator->errors()->first()
            ], 422);
        }

        DB::beginTransaction();

        try {
            $createdCount = 0;
            $updatedCount = 0;
            $skippedCount = 0;
            $processedIds = [];

            foreach ($request->marks as $markData) {
                if (!isset($markData['admission_id']) || !isset($markData['marks'])) {
                    continue;
                }

                // Skip duplicates in same request
                if (in_array($markData['admission_id'], $processedIds)) {
                    Log::warning('Duplicate admission_id in same request', [
                        'admission_id' => $markData['admission_id']
                    ]);
                    $skippedCount++;
                    continue;
                }
                $processedIds[] = $markData['admission_id'];

                // Build unique data array (all fields except marks)
                $uniqueData = [
                    'admission_id' => $markData['admission_id'],
                    'phase' => $request->phase,
                    'assessment_type' => $request->assessment_type,
                    'assessment_group' => $request->assessment_group,
                    'type' => $request->type,
                ];

                // Add nullable fields only if they have values
                $nullableFields = ['subject_id', 'code_name', 'exercises', 'bn', 'coy', 'pl', 'sec'];
                foreach ($nullableFields as $field) {
                    if ($request->filled($field)) {
                        $uniqueData[$field] = $request->$field;
                    }
                }

                // Handle week/month based on type
                if ($request->type === 'Weekly' && $request->filled('week')) {
                    $uniqueData['week'] = $request->week;
                } elseif ($request->type === 'Monthly' && $request->filled('month')) {
                    $uniqueData['month'] = $request->month;
                }

                try {
                    // Use the new updateOrCreateUnique method from our model
                    $assessment = Assessment::updateOrCreateUnique(
                        $uniqueData,  // Unique conditions
                        ['marks' => $markData['marks']]  // Only update marks
                    );

                    if ($assessment->wasRecentlyCreated) {
                        $createdCount++;
                        Log::info('Created assessment', [
                            'assessment_id' => $assessment->id,
                            'admission_id' => $markData['admission_id']
                        ]);
                    } else {
                        $updatedCount++;
                        Log::info('Updated assessment', [
                            'assessment_id' => $assessment->id,
                            'admission_id' => $markData['admission_id']
                        ]);
                    }
                } catch (\Illuminate\Validation\ValidationException $e) {
                    // This catches our custom duplicate validation
                    Log::warning('Duplicate assessment prevented', [
                        'admission_id' => $markData['admission_id'],
                        'message' => $e->getMessage()
                    ]);
                    $skippedCount++;
                }
            }

            DB::commit();

            $totalProcessed = count($processedIds);
            $message = "Marks saved successfully. ";
            
            $parts = [];
            if ($createdCount > 0) $parts[] = "$createdCount new";
            if ($updatedCount > 0) $parts[] = "$updatedCount updated";
            if ($skippedCount > 0) $parts[] = "$skippedCount skipped";
            
            if (!empty($parts)) {
                $message .= "(" . implode(', ', $parts) . " out of $totalProcessed students)";
            }

            return response()->json([
                'status' => 'success',
                'message' => $message,
                'data' => [
                    'created' => $createdCount,
                    'updated' => $updatedCount,
                    'skipped' => $skippedCount,
                    'total_processed' => $totalProcessed
                ]
            ]);

        } catch (\Illuminate\Database\QueryException $e) {
            DB::rollBack();
            
            // Check if it's a duplicate entry error
            if ($e->getCode() == 23000) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Duplicate entry detected. These marks may already exist in the system.'
                ], 422);
            }
            
            throw $e;
        }
    } catch (\Exception $e) {
        if (DB::transactionLevel() > 0) {
            DB::rollBack();
        }

        Log::error('Error saving marks', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);

        return response()->json([
            'status' => 'error',
            'message' => 'An error occurred while saving marks: ' . $e->getMessage()
        ], 500);
    }
}   
    

    public function printMarksSheet(Request $request)
{
    $data = [
        'school' => School::find($request->school_id)?->name,
        'course' => Course::find($request->course_id)?->name,
        'phase' => $request->choose_phase,
        'assessmentType' => $request->choose_assessment_type,
        'assessmentGroup' => $request->choose_assessment_group,
        'examType' => $request->choose_type,
        'bn' => $request->bn,
        'coy' => $request->coy,
        'pl' => $request->pl,
         'code_name' => $request->code_name,
        'sec' => $request->sec,
        'subject' => Subject::find($request->subject_id)?->code,
        'exercises' => $request->exercises,
        'week' => $request->weeks,
        'students' => $request->students ?? [],
    ];

    return view('marks.print', $data);
}

    public function marksSheet(){

         // Fetch the necessary data for dropdowns
         $schools = School::pluck('name', 'id');
         $courses = Course::pluck('name', 'id');
         $intakes = Intake::pluck('year', 'id');
         $subjects = Subject::pluck('name', 'id');

         return view('marks.marks_sheet', compact('schools', 'courses', 'intakes', 'subjects'));
    }







    /**
     * Calculate the weight based on assessment group, type and assessment type
     *
     * @param string $assessmentGroup
     * @param string $type
     * @param string $assessmentType
     * @return int
     */
    private function calculateWeight(string $assessmentGroup, string $type, string $assessmentType, string $exercises = null): int
    {
        if ($assessmentGroup === 'Theory') {
            // Get theory weights from ExamWeight table
            $weight = ExamWeight::where('name', $type)->first();
            return $weight ? $weight->weight : 0;
        }

        if ($assessmentGroup === 'Practical') {
            if ($exercises) {
                // Get exercise specific weights from Exercise table
                $exerciseWeight = Exercise::where('title', $exercises)->first();
                return $exerciseWeight ? $exerciseWeight->weight : 0;
            }

            // Fallback to assessment type weights from Exercise table
            $assessmentWeight = Exercise::where('title', $assessmentType)->first();
            return $assessmentWeight ? $assessmentWeight->weight : 0;
        }

        return 0;
    }




public function fetchStudents(Request $request)
{
    try {
        $validated = $request->validate([
            'school_id'               => 'nullable|integer|exists:schools,id',
            'course_id'               => 'nullable|integer|exists:courses,id',
            'intake_id'               => 'nullable|integer|exists:intakes,id',
            'bn'                      => 'nullable|string|max:100',
            'coy'                     => 'nullable|string|max:100',
            'pl'                      => 'nullable|string|max:100',
            'sec'                     => 'nullable|string|max:100',
        ]);

        // Log the incoming request
        \Log::info('fetchStudents request:', [
            'all' => $request->all(),
            'validated' => $validated
        ]);

        $query = Admission::query();

        // First, let's see what we have in the database for this school/course/intake
        if (!empty($validated['school_id']) && !empty($validated['course_id']) && !empty($validated['intake_id'])) {
            $countQuery = clone $query;
            $totalStudents = $countQuery
                ->where('school_id', $validated['school_id'])
                ->where('course_id', $validated['course_id'])
                ->where('intake_id', $validated['intake_id'])
                ->count();
            
            \Log::info('Total students for school/course/intake:', [
                'school_id' => $validated['school_id'],
                'course_id' => $validated['course_id'],
                'intake_id' => $validated['intake_id'],
                'count' => $totalStudents
            ]);

            // Get sample students to see their bn,coy,pl,sec values
            $samples = Admission::where('school_id', $validated['school_id'])
                ->where('course_id', $validated['course_id'])
                ->where('intake_id', $validated['intake_id'])
                ->limit(5)
                ->get(['id', 'army_number', 'name', 'bn', 'coy', 'pl', 'sec']);
            
            \Log::info('Sample students values:', $samples->toArray());
        }

        $exactFilters = ['school_id', 'course_id', 'intake_id', 'bn', 'coy', 'pl', 'sec'];

        foreach ($exactFilters as $filter) {
            if (!empty($validated[$filter])) {
                $value = $validated[$filter];
                \Log::info("Applying filter: {$filter} = '{$value}'");
                $query->where($filter, $value);
            }
        }

        \Log::info('Final query:', [
            'sql' => $query->toSql(),
            'bindings' => $query->getBindings()
        ]);

        $students = $query
            ->orderBy('name')
            ->limit(1000)
            ->get(['id', 'army_number', 'name', 'rank', 'is_decommissioned', 'bn', 'coy', 'pl', 'sec']);

        \Log::info('Students found:', ['count' => $students->count()]);

        return response()->json($students);

    } catch (\Exception $e) {
        \Log::error('fetchStudents error: ' . $e->getMessage(), [
            'trace' => $e->getTraceAsString(),
            'request' => $request->all()
        ]);
        return response()->json(['message' => 'Error fetching students.'], 500);
    }
}




}
