<?php

namespace App\Http\Controllers;

use App\Exports\AssessmentsExport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use App\Models\Course;
use App\Models\Intake;
use App\Models\Assessment;
use App\Models\Subject;
use App\Models\School;
use App\Models\Exercise;
use App\Models\Admission;
use App\Models\ExamWeight;
use App\Models\Phase;
use App\Models\Bn;
use App\Models\Coy;
use App\Models\Pl;
use App\Models\Sec;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Exports\AdmissionTemplateExport;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Exports\MarksExport;


class MarksController extends Controller
{

public function exportExcel(Request $request)
{
    return Excel::download(new MarksExport($request), 'marks.xlsx');
}


    public function exportPdf(Request $request)
    {
        $data = $this->getFilteredData($request);

        $groupedData = $data->groupBy(function ($item) {
            return $item->admission->army_number . '|' . $item->admission->rank . '|' . $item->admission->name . '|' . $item->admission->sex;
        });

        $pdf = PDF::loadView('exports.marks-pdf', [
            'groupedMarks' => $groupedData
        ]);

        return $pdf->stream
        ('marks.pdf');
    }

    private function getFilteredData(Request $request)
    {
        $query = Assessment::with(['admission', 'subject']);

        if ($request->filled('search')) {
            $query->whereHas('admission', function ($query) use ($request) {
                $query->where('name', 'like', "%{$request->search}%")
                    ->orWhere('army_number', 'like', "%{$request->search}%");
            });
        }


        if ($request->assessment_type) {
            $query->where('assessment_type', $request->assessment_type);
        }

        if ($request->assessment_type) {
            $query->where('month', $request->month);
        }

        if ($request->assessment_group) {
            $query->where('assessment_group', $request->assessment_group);
        }

        if ($request->type) {
            $query->where('type', $request->type);
        }

        if ($request->week) {
            $query->where('week', $request->week);
        }

        if ($request->subject_id) {
            $query->where('subject_id', $request->subject_id);
        }

        if ($request->exercises) {
            $query->where('exercises', $request->exercises);
        }

        if ($request->bn) {
            $query->where('bn', $request->bn);
        }

        if ($request->coy) {
            $query->where('coy', $request->coy);
        }

        return $query->orderBy('marks', 'desc')->get();
    }






public function generateReport(Request $request)
{
    // Build the query with filters
    $query = Assessment::query();


    // Search filter
    if ($request->filled('search')) {
        $query->whereHas('admission', function ($query) use ($request) {
            $query->where('name', 'like', "%{$request->search}%")
                ->orWhere('army_number', 'like', "%{$request->search}%")
                ->orWhere('rank', 'like', "%{$request->search}%");
        });
    }

    // School filter
    if ($request->filled('school_id')) {
        $query->whereHas('admission', function ($query) use ($request) {
            $query->where('school_id', $request->school_id);
        });
    }

    // Course filter
    if ($request->filled('course_id')) {
        $query->whereHas('admission', function ($query) use ($request) {
            $query->where('course_id', $request->course_id);
        });
    }

    // Intake filter
    if ($request->filled('intake_id')) {
        $query->whereHas('admission', function ($query) use ($request) {
            $query->where('intake_id', $request->intake_id);
        });
    }

    // Phase filter
    if ($request->filled('choose_phase')) {
        $query->whereHas('admission', function ($query) use ($request) {
            $query->where('phase', $request->choose_phase);
        });
    }

    // Assessment Type filter
    if ($request->filled('choose_assessment_type')) {
        $query->where('assessment_type', $request->choose_assessment_type);
    }

    // Assessment Group filter
    if ($request->filled('choose_assessment_group')) {
        $query->where('assessment_group', $request->choose_assessment_group);
    }

    // Exam Week filter
    if ($request->filled('weeks')) {
        $query->where('exam_week', $request->weeks);
    }

    // Subject filter
    if ($request->filled('subject_id')) {
        $query->where('subject_id', $request->subject_id);
    }

    // Other filters
    if ($request->filled('bn')) {
        $query->whereHas('admission', function ($query) use ($request) {
            $query->where('bn', $request->bn);
        });
    }

    if ($request->filled('coy')) {
        $query->whereHas('admission', function ($query) use ($request) {
            $query->where('coy', $request->coy);
        });
    }

    if ($request->filled('pl')) {
        $query->whereHas('admission', function ($query) use ($request) {
            $query->where('pl', $request->pl);
        });
    }

    if ($request->filled('sec')) {
        $query->whereHas('admission', function ($query) use ($request) {
            $query->where('sec', $request->sec);
        });
    }

    // Get the filtered assessments
    $assessments = $query->with('admission', 'subject')->get();

        // Retrieve schools data
        $schools = School::all()->pluck('name', 'id'); // Adjust this query based on your actual data structure


    // If download PDF requested
    if ($request->has('download_pdf') && $request->download_pdf) {
        $pdf = PDF::loadView('reports.marks_report', compact('assessments'));
        return $pdf->download('marks_report.pdf');
    }

    // Return the report view
   return view('reports.assessment_report', compact('assessments', 'schools'));

}


public function index2(Request $request)
{

    
 
    // Default page size
    $perPage = 25;
    $page = $request->get('page', 1);


    $phase = Phase::where('is_active', 1)->first();

    // Build base query with proper joins
    $query = Assessment::query()
        ->select(
            'assessments.*',
            'admissions.army_number',
            'admissions.rank',
            'admissions.name'
        )
        ->where('assessments.phase', $phase->name)
        ->join('admissions', 'assessments.admission_id', '=', 'admissions.id');
    

    // Apply search if provided
    if ($request->has('search')) {
        $searchTerm = $request->search;
        $query->where(function($q) use ($searchTerm) {
            $q->where('admissions.name', 'LIKE', "%{$searchTerm}%")
              ->orWhere('admissions.army_number', 'LIKE', "%{$searchTerm}%")
              ->orWhere('admissions.rank', 'LIKE', "%{$searchTerm}%");
        });
    }

    // Apply filters if provided
    foreach (['assessment_type', 'assessment_group', 'type', 'weeks', 'subject_id', 'exercises', 'bn', 'coy', 'pl', 'sec'] as $filter) {
        if ($request->filled($filter)) {
            $query->where("assessments.{$filter}", $request->$filter);
        }
    }

    // Get the filtered and paginated results
    $groupedResults = $query->get()->groupBy('admission_id');
    $total = $groupedResults->count();
    $groupedResults = $groupedResults->slice(($page - 1) * $perPage, $perPage);

    // Create paginator
    $marks = new \Illuminate\Pagination\LengthAwarePaginator(
        $groupedResults,
        $total,
        $perPage,
        $page,
        ['path' => $request->url(), 'query' => $request->query()]
    );

    // Load subjects for the filter dropdown
    $subjects = Subject::all();

        // Load data for dropdowns
        $subjects = Subject::all();
        $bn = Bn::pluck('name');
        $coy = Coy::pluck('name');
        $sec = Sec::pluck('name');
        $pl = Pl::pluck('name');
        $exercise = Exercise::get();



        return view('marks.index2', compact(
            'marks',
            'subjects',
            'bn',
            'coy',
            'sec',
            'pl',
            'exercise'
        ));

    // return view('marks.index2', compact('marks', 'subjects'));
}




public function index(Request $request)
{
    $perPage = $request->get('per_page', 25);
    $page = $request->get('page', 1);

    // Add sorting parameters
    $sortField = $request->input('sort', 'admissions.army_number');
    $sortDirection = $request->input('direction', 'asc');
    $sortWeek = $request->input('week');
    
    // Expanded allowed sortable fields
    $allowedSorts = [
        'admissions.army_number',
        'admissions.rank',
        'admissions.name',
        'assessments.marks'
    ];

    // Cache the active phase - this is the current phase
    $phase = Cache::remember('active_phase', 3600, function () {
        return Phase::where('is_active', 1)->first();
    });

    if (!$phase) {
        abort(404, 'No active phase found');
    }

    // STEP 1: Build a subquery to get distinct admission_ids that match filters
    // ONLY for the current/active phase
    $admissionIdsQuery = Assessment::query()
        ->select('assessments.admission_id')
        ->where('assessments.phase', $phase->name)  // CRITICAL: Filter by current phase
        ->join('admissions', function ($join) {
            $join->on('assessments.admission_id', '=', 'admissions.id')
                 ->whereNull('admissions.deleted_at');
        });

    // Apply search
    if ($request->filled('search')) {
        $searchTerm = $request->search;
        $admissionIdsQuery->where(function($q) use ($searchTerm) {
            $q->where('admissions.name', 'LIKE', "%{$searchTerm}%")
                ->orWhere('admissions.army_number', 'LIKE', "%{$searchTerm}%")
                ->orWhere('admissions.rank', 'LIKE', "%{$searchTerm}%");
        });
    }

    // Apply filters
    foreach (['assessment_type', 'assessment_group', 'type', 'subject_id', 'exercises', 'bn', 'coy', 'pl', 'sec', 'month'] as $filter) {
        if ($request->filled($filter)) {
            $admissionIdsQuery->where("assessments.{$filter}", $request->$filter);
        }
    }

    // Handle week filter
    if ($request->filled('week')) {
        $admissionIdsQuery->where('assessments.week', $request->week);
    }

    // STEP 2: Get sorted and paginated admission IDs
    if ($sortField === 'assessments.marks') {
        // For marks sorting, get the specific mark value
        $admissionIdsQuery->when($sortWeek, function($q) use($sortWeek) {
            $q->where('assessments.week', $sortWeek);
        });
        $admissionIdsQuery->orderBy('assessments.marks', $sortDirection);
    } elseif (in_array($sortField, $allowedSorts)) {
        $admissionIdsQuery->orderBy($sortField, $sortDirection);
    }
    
    $admissionIdsQuery->orderBy('admissions.army_number', 'asc');

    // Get distinct admission IDs
    $allAdmissionIds = $admissionIdsQuery
        ->distinct()
        ->pluck('assessments.admission_id');

    $total = $allAdmissionIds->count();
    $paginatedAdmissionIds = $allAdmissionIds->slice(($page - 1) * $perPage, $perPage)->values();

    // STEP 3: Fetch all assessments for paginated admission IDs
    // ONLY from the current/active phase
    $query = Assessment::query()
        ->select(
            'assessments.*',
            'admissions.army_number',
            'admissions.rank',
            'admissions.name'
        )
        ->where('assessments.phase', $phase->name)  // CRITICAL: Filter by current phase
        ->whereIn('assessments.admission_id', $paginatedAdmissionIds)
        ->join('admissions', function ($join) {
            $join->on('assessments.admission_id', '=', 'admissions.id')
                 ->whereNull('admissions.deleted_at');
        });

    // Apply the same filters to main query
    foreach (['assessment_type', 'assessment_group', 'type', 'subject_id', 'exercises', 'bn', 'coy', 'pl', 'sec', 'month'] as $filter) {
        if ($request->filled($filter)) {
            $query->where("assessments.{$filter}", $request->$filter);
        }
    }

    if ($request->filled('week')) {
        $query->where('assessments.week', $request->week);
    }

    // Get results and group by admission_id
    $results = $query->get();
    
    // Group and maintain order based on paginated IDs
    $groupedResults = collect();
    foreach ($paginatedAdmissionIds as $admissionId) {
        $groupedResults[$admissionId] = $results->where('admission_id', $admissionId);
    }

    // Add row numbers
    $offset = ($page - 1) * $perPage;
    $groupedResults = $groupedResults->map(function ($group) use (&$offset) {
        $group->row_number = $offset + 1;
        $offset++;
        return $group;
    });

    // Create paginator
    $marks = new \Illuminate\Pagination\LengthAwarePaginator(
        $groupedResults,
        $total,
        $perPage,
        $page,
        [
            'path' => $request->url(),
            'query' => $request->query()
        ]
    );

    // Load data for dropdowns - optimized
    $subjects = Subject::select('id', 'name')->get();
    $bn = Bn::pluck('name');
    $coy = Coy::pluck('name');
    $sec = Sec::pluck('name');
    $pl = Pl::pluck('name');
    
    // Load exercises ONLY for current phase
    $exercise = Exercise::select('id', 'title', 'weight')
        ->where('phase_id', $phase->id)  // Filter exercises by current phase
        ->get();

    // Prepare data for view
    $viewData = compact(
        'phase',
        'marks',
        'subjects',
        'bn',
        'coy',
        'sec',
        'pl',
        'exercise',
        'sortField',
        'sortDirection',
        'sortWeek'
    );

    return view('marks.index', $viewData);
}

// Method to clear cache when assessments are updated
public function clearAssessmentsCache()
{
    // // Clear all assessment-related cache keys
    // $cacheKeys = Cache::getRedis()->keys('assessments_index_*');
    // foreach ($cacheKeys as $key) {
    //     Cache::forget(str_replace(config('cache.prefix') . ':', '', $key));
    // }
}


    public function edit($id)
{
    $assessment = Assessment::findOrFail($id);
    return view('marks.edit', compact('assessment'));
}

public function update(Request $request, $id)
{
    $request->validate([
        'marks' => 'required|numeric|min:0|max:100',

    ]);

    $assessment = Assessment::findOrFail($id);
    $assessment->update([
        'marks' => $request->input('marks'),

    ]);

    return redirect()->route('marks.index')->with('success', 'Assessment updated successfully');
}

public function destroy($id)
{
    $assessment = Assessment::findOrFail($id);
    $assessment->delete();

    return redirect()->route('marks.index')->with('success', 'Assessment deleted successfully');
}
public function bulkDelete(Request $request)
{
    if (!Auth::user()->type == 'admin') {
        return redirect()->route('marks.index2')
            ->with('error', 'Unauthorized action.');
    }

    $selectedIds = $request->input('selected_ids', []);

    if (empty($selectedIds)) {
        return redirect()->route('marks.index2')
            ->with('error', 'No items selected for deletion.');
    }

    try {
        Assessment::whereIn('id', $selectedIds)->delete();

        return redirect()->route('marks.index2')
            ->with('success', count($selectedIds) . ' assessments deleted successfully.');
    } catch (\Exception $e) {
        return redirect()->route('marks.index2')
            ->with('error', 'Error deleting assessments. Please try again.');
    }
}
public function viewMoreMarks($id)
{
    // Get student admission details
    $admission = Admission::findOrFail($id);

$currentPhase  = Phase::where('is_active', 1)->first();

    // Get all assessments for the student
    $assessments = Assessment::where('admission_id', $id)
      ->where('phase',$currentPhase->name)
        ->with(['subject'])
        ->get()
        ->groupBy(['assessment_group', 'type']);

    // Get weights from models
    $exerciseWeights = Exercise::pluck('weight', 'title')->toArray();
    $examWeights = ExamWeight::pluck('weight', 'name')->toArray();

    return view('marks.viewmore', compact('admission', 'assessments', 'exerciseWeights', 'examWeights'));
}

public function editMarks()
    {
        $schools = School::pluck('name', 'id');
        $courses = Course::pluck('name', 'id');
        $intakes = Intake::pluck('year', 'id');
        $phases = Phase::all();
        $subjects = Subject::all();
        
        // Get distinct values for filters
        $bn = Assessment::whereNotNull('bn')->distinct()->pluck('bn');
        $coy = Assessment::whereNotNull('coy')->distinct()->pluck('coy');
        
        // Get exercises
        $exercises = Assessment::whereNotNull('exercises')->distinct()->pluck('exercises');
        
        return view('marks.edit', compact(
            'schools', 
            'courses', 
            'intakes', 
            'phases', 
            'subjects',
            'bn',
            'coy',
            'exercises'
        ));
    }

    /**
     * Get filtered marks for editing
     */
    public function getFilteredMarks(Request $request)
    {
        $query = Assessment::with(['admission' => function($q) {
            $q->select('id', 'admission_id', 'name', 'army_number', 'rank', 
                      'bn', 'coy', 'pl', 'sec', 'is_decommissioned', 'school_id', 'course_id', 'intake_id');
        }, 'subject']);

        // Apply filters
        if ($request->filled('school_id')) {
            $query->whereHas('admission', function($q) use ($request) {
                $q->where('school_id', $request->school_id);
            });
        }
        
        if ($request->filled('course_id')) {
            $query->whereHas('admission', function($q) use ($request) {
                $q->where('course_id', $request->course_id);
            });
        }
        
        if ($request->filled('intake_id')) {
            $query->whereHas('admission', function($q) use ($request) {
                $q->where('intake_id', $request->intake_id);
            });
        }
        
        if ($request->filled('phase')) {
            $query->where('phase', $request->phase);
        }
        
        if ($request->filled('assessment_type')) {
            $query->where('assessment_type', $request->assessment_type);
        }
        
        if ($request->filled('assessment_group')) {
            $query->where('assessment_group', $request->assessment_group);
        }
        
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }
        
        if ($request->filled('subject_id')) {
            $query->where('subject_id', $request->subject_id);
        }
        
        if ($request->filled('exercises')) {
            $query->where('exercises', $request->exercises);
        }
        
        if ($request->filled('week')) {
            $query->where('week', $request->week);
        }
        
        if ($request->filled('month')) {
            $query->where('month', $request->month);
        }

        if ($request->filled('code_name')) {
            $query->where('code_name', $request->code_name);
        }

        // Filter by bn, coy, pl, sec
        if ($request->filled('bn')) {
            $query->where('bn', $request->bn);
        }
        
        if ($request->filled('coy')) {
            $query->where('coy', $request->coy);
        }
        
        if ($request->filled('pl')) {
            $query->where('pl', $request->pl);
        }
        
        if ($request->filled('sec')) {
            $query->where('sec', $request->sec);
        }

        $assessments = $query->get()->map(function($assessment) {
            return [
                'id' => $assessment->id,
                'admission_id' => $assessment->admission_id,
                'marks' => $assessment->marks,
                'army_number' => $assessment->admission->army_number ?? 'N/A',
                'rank' => $assessment->admission->rank ?? 'N/A',
                'name' => $assessment->admission->name ?? 'N/A',
                'bn' => $assessment->bn ?? 'N/A',
                'coy' => $assessment->coy ?? 'N/A',
                'pl' => $assessment->pl ?? 'N/A',
                'sec' => $assessment->sec ?? 'N/A',
                'subject' => $assessment->subject->name ?? 'N/A',
                'exercises' => $assessment->exercises ?? 'N/A',
                'week' => $assessment->week ?? 'N/A',
                'month' => $assessment->month ?? 'N/A',
                'type' => $assessment->type ?? 'N/A',
                'is_decommissioned' => $assessment->admission->is_decommissioned ?? false
            ];
        });

        return response()->json($assessments);
    }

    /**
     * Get marks for a single student by army number
     */
    public function getStudentMarksByArmyNumber(Request $request)
    {
        $armyNumber = $request->input('army_number');
        
        if (!$armyNumber) {
            return response()->json(['error' => 'Army number is required'], 400);
        }

        // Find admission by army number
        $admission = Admission::with(['school', 'course', 'intake'])
            ->where('army_number', $armyNumber)
            ->first();
        
        if (!$admission) {
            return response()->json(['error' => 'Student not found'], 404);
        }

        $assessments = Assessment::with(['subject'])
            ->where('admission_id', $admission->id)
            ->orderBy('phase')
            ->orderBy('assessment_type')
            ->orderBy('type')
            ->get()
            ->map(function($assessment) use ($admission) {
                return [
                    'id' => $assessment->id,
                    'admission_id' => $assessment->admission_id,
                    'marks' => $assessment->marks,
                    'phase' => $assessment->phase,
                    'assessment_type' => $assessment->assessment_type,
                    'assessment_group' => $assessment->assessment_group,
                    'type' => $assessment->type,
                    'subject' => $assessment->subject->name ?? 'N/A',
                    'subject_code' => $assessment->subject->code ?? 'N/A',
                    'exercises' => $assessment->exercises,
                    'code_name' => $assessment->code_name,
                    'week' => $assessment->week,
                    'month' => $assessment->month,
                    'bn' => $assessment->bn,
                    'coy' => $assessment->coy,
                    'pl' => $assessment->pl,
                    'sec' => $assessment->sec,
                    'school' => $admission->school->name ?? 'N/A',
                    'course' => $admission->course->name ?? 'N/A',
                    'intake' => $admission->intake->year ?? 'N/A',
                    'student_info' => [
                        'admission_id' => $admission->admission_id,
                        'army_number' => $admission->army_number ?? 'N/A',
                        'rank' => $admission->rank ?? 'N/A',
                        'name' => $admission->name ?? 'N/A',
                        'is_decommissioned' => $admission->is_decommissioned ?? false
                    ]
                ];
            });

        return response()->json($assessments);
    }

    /**
     * Get marks for a single student by admission ID
     */
    public function getStudentMarks($admissionId)
    {
        // Get the admission record first
        $admission = Admission::with(['school', 'course', 'intake'])->find($admissionId);
        
        if (!$admission) {
            return response()->json(['error' => 'Student not found'], 404);
        }

        $assessments = Assessment::with(['subject'])
            ->where('admission_id', $admissionId)
            ->orderBy('phase')
            ->orderBy('assessment_type')
            ->get()
            ->map(function($assessment) use ($admission) {
                return [
                    'id' => $assessment->id,
                    'admission_id' => $assessment->admission_id,
                    'marks' => $assessment->marks,
                    'phase' => $assessment->phase,
                    'assessment_type' => $assessment->assessment_type,
                    'assessment_group' => $assessment->assessment_group,
                    'type' => $assessment->type,
                    'subject' => $assessment->subject->name ?? 'N/A',
                    'subject_code' => $assessment->subject->code ?? 'N/A',
                    'exercises' => $assessment->exercises,
                    'code_name' => $assessment->code_name,
                    'week' => $assessment->week,
                    'month' => $assessment->month,
                    'bn' => $assessment->bn,
                    'coy' => $assessment->coy,
                    'pl' => $assessment->pl,
                    'sec' => $assessment->sec,
                    'school' => $admission->school->name ?? 'N/A',
                    'course' => $admission->course->name ?? 'N/A',
                    'intake' => $admission->intake->year ?? 'N/A',
                    'student_info' => [
                        'admission_id' => $admission->admission_id,
                        'army_number' => $admission->army_number ?? 'N/A',
                        'rank' => $admission->rank ?? 'N/A',
                        'name' => $admission->name ?? 'N/A',
                        'is_decommissioned' => $admission->is_decommissioned ?? false
                    ]
                ];
            });

        return response()->json($assessments);
    }

    /**
     * Update a single mark record with organizational and time details
     */
    public function updateMark(Request $request, $id)
    {
        $request->validate([
            'marks' => 'nullable|numeric|min:0|max:100',
            'bn' => 'nullable|string',
            'coy' => 'nullable|string',
            'pl' => 'nullable|string',
            'sec' => 'nullable|string',
            'week' => 'nullable|string',
            'month' => 'nullable|string'
        ]);

        $assessment = Assessment::findOrFail($id);
        
        // Check if student is decommissioned
        if ($assessment->admission && $assessment->admission->is_decommissioned) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot update marks for decommissioned cadet'
            ], 403);
        }

        // Update only provided fields
        if ($request->has('marks')) {
            $assessment->marks = $request->marks;
        }
        if ($request->has('bn')) {
            $assessment->bn = $request->bn;
        }
        if ($request->has('coy')) {
            $assessment->coy = $request->coy;
        }
        if ($request->has('pl')) {
            $assessment->pl = $request->pl;
        }
        if ($request->has('sec')) {
            $assessment->sec = $request->sec;
        }
        if ($request->has('week')) {
            $assessment->week = $request->week;
        }
        if ($request->has('month')) {
            $assessment->month = $request->month;
        }

        $assessment->save();

        return response()->json([
            'success' => true,
            'message' => 'Assessment updated successfully',
            'data' => $assessment
        ]);
    }

    /**
     * Bulk update marks
     */
    public function bulkUpdateMarks(Request $request)
    {
        $request->validate([
            'marks' => 'required|array',
            'marks.*.id' => 'required|exists:assessments,id',
            'marks.*.marks' => 'required|numeric|min:0|max:100'
        ]);

        $updated = 0;
        $failed = 0;
        
        DB::beginTransaction();
        
        try {
            foreach ($request->marks as $markData) {
                $assessment = Assessment::with('admission')->find($markData['id']);
                
                if ($assessment && (!$assessment->admission || !$assessment->admission->is_decommissioned)) {
                    $assessment->marks = $markData['marks'];
                    $assessment->save();
                    $updated++;
                } else {
                    $failed++;
                }
            }
            
            DB::commit();
            
            $message = "{$updated} marks updated successfully";
            if ($failed > 0) {
                $message .= ", {$failed} skipped (decommissioned cadets)";
            }

            return response()->json([
                'success' => true,
                'message' => $message,
                'updated' => $updated,
                'failed' => $failed
            ]);
            
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error updating marks: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete an assessment record
     */
    public function deleteMark($id)
    {
        try {
            $assessment = Assessment::findOrFail($id);
            $assessment->delete();

            return response()->json([
                'success' => true,
                'message' => 'Mark deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting mark: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get distinct organizational units for dropdowns
     */
    public function getOrganizationalUnits()
    {
        $bn = Assessment::whereNotNull('bn')
            ->distinct()
            ->pluck('bn')
            ->filter()
            ->values();
            
        $coy = Assessment::whereNotNull('coy')
            ->distinct()
            ->pluck('coy')
            ->filter()
            ->values();

        return response()->json([
            'bn' => $bn,
            'coy' => $coy
        ]);
    }

    /**
     * Get statistics for filtered marks
     */
    public function getMarkStats(Request $request)
    {
        $query = Assessment::query();

        // Apply same filters as getFilteredMarks
        if ($request->filled('phase')) {
            $query->where('phase', $request->phase);
        }
        
        if ($request->filled('assessment_type')) {
            $query->where('assessment_type', $request->assessment_type);
        }
        
        if ($request->filled('bn')) {
            $query->where('bn', $request->bn);
        }

        $stats = [
            'total_students' => $query->distinct('admission_id')->count('admission_id'),
            'average_mark' => round($query->avg('marks'), 2),
            'highest_mark' => $query->max('marks'),
            'lowest_mark' => $query->min('marks'),
            'pass_rate' => round(($query->where('marks', '>=', 50)->count() / max($query->count(), 1)) * 100, 2)
        ];

        return response()->json($stats);
    }
}
