<?php

namespace App\Http\Controllers;

use App\Models\Phase;
use App\Models\Admission;
use App\Models\GpvReportDetail;
use App\Models\GpvReport;
use App\Models\Intake;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Validator;   

class PhaseController extends Controller
{
    public function index()
    {
        $phase = Phase::all();
        return view('phase.index', compact('phase'));
    }
/**
     * Start a phase by deactivating all other phases and activating the selected one.
     *
     * @param Phase $phase The phase to activate
     * @return \Illuminate\Http\RedirectResponse
     */
    public function start(Phase $phase)
    {
        // First deactivate all phases in the same intake
        Phase::where('intake_id', $phase->intake_id)
             ->update(['is_active' => 0]);
        
        // Then activate only the selected phase
        $phase->update(['is_active' => 1]);
        
  
        
        return redirect()->back()->with('success', 'Phase started successfully.');
    }

    /**
     * End a phase by deactivating it.
     *
     * @param Phase $phase The phase to deactivate
     * @return \Illuminate\Http\RedirectResponse
     */
    public function end(Phase $phase)
    {
        $phase->update(['is_active' => 0]);
        
        return redirect()->back()->with('success', 'Phase ended successfully.');
    }

    /**
     * Move to the next phase sequentially.
     * 
     * @param int $intakeId The intake ID
     * @return \Illuminate\Http\RedirectResponse
     */
 public function nextPhase($intakeId)
    {
        DB::beginTransaction();
        try {
            // Get the current active phase
            $currentPhase = Phase::where('intake_id', $intakeId)
                                 ->where('is_active', 1)
                                 ->first();

            if (!$currentPhase) {
                // If no active phase, start the first one (no archival needed)
                $firstPhase = Phase::where('intake_id', $intakeId)
                                   ->orderBy('id', 'asc')
                                   ->first();

                if ($firstPhase) {
                    DB::table('phase_configs')->where('id', $firstPhase->id)->update(['is_active' => 1]);
                    $firstPhase->is_active = 1;
                    $firstPhase->save();

                    DB::commit();
                    return redirect()->back()->with('success', 'First phase activated.');
                }

                DB::rollBack();
                return redirect()->back()->with('error', 'No phases found for this intake.');
            }

            // Find the next phase
            $nextPhase = Phase::where('intake_id', $intakeId)
                              ->where('id', '>', $currentPhase->id)
                              ->orderBy('id', 'asc')
                              ->first();

            if (!$nextPhase) {
                DB::rollBack();
                return redirect()->back()->with('info', 'Already at the last phase.');
            }

            // --- ARCHIVE THE CURRENT PHASE'S DATA BEFORE SWITCHING ---
            $this->archivePhaseData($currentPhase, $nextPhase);

            // Deactivate current phase in phase_configs and Phase model
            DB::table('phase_configs')->where('id', $currentPhase->id)->update(['is_active' => 0]);
            $currentPhase->is_active = 0;
            $currentPhase->save();

            // Activate next phase in phase_configs and Phase model
            DB::table('phase_configs')->where('id', $nextPhase->id)->update(['is_active' => 1]);
            $nextPhase->is_active = 1;
            $nextPhase->save();

            DB::commit();
            return redirect()->back()->with('success', 'Moved to next phase. Phase data archived.');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Failed to move to next phase: ' . $e->getMessage());
        }
    }

    /**
     * Archive all admissions of the current intake into a GpvReport,
     * tagged with the ending phase.
     *
     * @param Phase $currentPhase
     * @param Phase $nextPhase
     * @return void
     */


/**
 * Archive all admissions of the given old phase's intake into a GpvReport.
 *
 * @param Phase $oldPhase  The phase that is ending
 * @param Phase $newPhase  The phase that will become active
 * @return void
 */
private function archivePhaseData(Phase $oldPhase, Phase $newPhase)
{
    // Check if the intake exists before trying to use it
    $intakeExists = Intake::where('id', $oldPhase->intake_id)->exists();
    
    // Fetch all admissions belonging to the same intake as the old phase
    $admissions = Admission::where('intake_id', $oldPhase->intake_id)
                           ->with('student', 'course', 'school')
                           ->get();

    $studentsData = [];
    foreach ($admissions as $admission) {
        $studentsData[] = [
            'student_id'       => $admission->id,
            'admission_number' => $admission->admission_number ?? '',
            'army_number'      => optional($admission->student)->army_number ?? '',
            'rank'             => optional($admission->student)->rank ?? '',
            'name'             => optional($admission->student)->name ?? '',
            'school'           => optional($admission->school)->name ?? '',
            'course'           => optional($admission->course)->name ?? '',
            'theory'           => $admission->theory_score ?? 0,
            'practical'        => $admission->practical_score ?? 0,
            'cr_22'            => $admission->cr_22_score ?? 0,
            'pr'               => $admission->pr_score ?? 0,
            'total'            => $admission->total_score ?? 0,
            'grade'            => $admission->grade ?? '',
            'comment'          => $admission->comment ?? '',
            'remarks'          => $admission->remarks ?? '',
        ];
    }

    // Create a GpvReport record for this phase transition
    // Set intake_id to NULL if the intake doesn't exist
    $archiveReport = GpvReport::create([
        'report_name'    => 'Phase_Archive_' . $oldPhase->name . '_to_' . $newPhase->name . '_' . now()->format('Y-m-d_H-i-s'),
        'intake_id'      => $intakeExists ? $oldPhase->intake_id : null, // Set to NULL if intake doesn't exist
        'school_id'      => null,
        'course_id'      => null,
        'phase'          => $oldPhase->name,
        'bn'             => null,
        'coy'            => null,
        'pl'             => null,
        'grade'          => null,
        'army_number'    => null,
        'total_students' => count($studentsData),
        'filters'        => json_encode([
            'archive_type'       => 'phase_transition',
            'from_phase_id'      => $oldPhase->id,
            'to_phase_id'        => $newPhase->id,
            'intake_id'          => $oldPhase->intake_id,
            'intake_exists'      => $intakeExists,
            'original_intake_id' => $oldPhase->intake_id // Store the original ID in filters even if it doesn't exist
        ]),
        'created_by'     => auth()->id(), // This should exist in your users table
    ]);

    // Insert each student as a detail record (bulk insert for efficiency)
    if (!empty($studentsData)) {
        $details = [];
        foreach ($studentsData as $student) {
            $details[] = [
                'gpv_report_id'    => $archiveReport->id,
                'admission_id'     => $student['student_id'],
                'admission_number' => $student['admission_number'],
                'army_number'      => $student['army_number'],
                'rank'             => $student['rank'],
                'name'             => $student['name'],
                'school'           => $student['school'],
                'course'           => $student['course'],
                'theory_score'     => $student['theory'],
                'practical_score'  => $student['practical'],
                'cr_22_score'      => $student['cr_22'],
                'pr_score'         => $student['pr'],
                'total_score'      => $student['total'],
                'grade'            => $student['grade'],
                'comment'          => $student['comment'],
                'remarks'          => $student['remarks'],
                'created_at'       => now(),
                'updated_at'       => now(),
            ];
        }
        GpvReportDetail::insert($details);
    }
}


    public function create()
    {
        return view('phase.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'weeks' => 'required|numeric',
        ]);

        Phase::create($request->all());

        return redirect()->route('wss')->with('success', 'Phase created successfully.');
    }

    public function edit(Phase $phase)
    {
        return view('phase.edit', compact('phase'));
    }

    public function cres(Phase $phase)
    {
        return view('phase.cres', compact('phase'));
    }

    public function pr(Phase $phase)
    {
        return view('phase.pr', compact('phase'));
    }

    public function update(Request $request, Phase $phase)
    {
        $request->validate([
            'name' => 'string|max:255',
         'weeks' => 'numeric',
            'cres_weight' => 'numeric',
            'pr' => 'numeric'
        ]);

        $phase->update($request->all());

        return redirect()->route('wss')->with('success', 'Phase updated successfully.');
    }

    public function destroy(Phase $phase)
    {
        $phase->delete();

        return redirect()->route('wss')->with('success', 'Phase deleted successfully.');
    }
}
