<?php

namespace App\Http\Controllers;

use App\Models\RemarkTemplate;
use App\Models\Phase;
use Illuminate\Http\Request;

class RemarkTemplateController extends Controller
{
    protected array $grades = ['A', 'B', 'C', 'D', 'E', 'F'];
    protected array $types  = ['PL_COMD', 'OC', 'CI', 'COMDT'];

    /**
     * Display all templates grouped by grade, filtered by phase
     */
    public function index(Request $request)
    {
        $phases          = Phase::orderBy('name')->get();
        $selectedPhaseId = $request->input('phase_id', optional(Phase::where('is_active', 1)->first())->id);

        $templates = RemarkTemplate::where('phase_id', $selectedPhaseId)
            ->orderByRaw("FIELD(grade, 'A','B','C','D','E','F')")
            ->orderByRaw("FIELD(type, 'PL_COMD','OC','CI','COMDT')")
            ->get()
            ->groupBy('grade');

        return view('remark-templates.index', compact('phases', 'templates', 'selectedPhaseId'));
    }

    /**
     * Show create form
     */
    public function create()
    {
        $phases = Phase::orderBy('name')->get();
        return view('remark-templates.create', [
            'phases' => $phases,
            'grades' => $this->grades,
            'types'  => $this->types,
        ]);
    }

    /**
     * Store — uses updateOrCreate so duplicate phase/grade/type overwrites cleanly
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'phase_id'    => 'required|exists:phases,id',
            'grade'       => 'required|in:A,B,C,D,E,F',
            'type'        => 'required|in:PL_COMD,OC,CI,COMDT',
            'remark_text' => 'required|string',
        ]);

        RemarkTemplate::updateOrCreate(
            [
                'phase_id' => $validated['phase_id'],
                'grade'    => $validated['grade'],
                'type'     => $validated['type'],
            ],
            ['remark_text' => $validated['remark_text']]
        );

        return redirect()
            ->route('remark-templates.index', ['phase_id' => $validated['phase_id']])
            ->with('success', "Remark for Grade {$validated['grade']} / {$validated['type']} saved.");
    }

    /**
     * Show a single template
     */
    public function show(RemarkTemplate $remarkTemplate)
    {
        return view('remark-templates.show', compact('remarkTemplate'));
    }

    /**
     * Show edit form
     */
    public function edit(RemarkTemplate $remarkTemplate)
    {
        $phases = Phase::orderBy('name')->get();
        return view('remark-templates.edit', [
            'remarkTemplate' => $remarkTemplate,
            'phases'         => $phases,
            'grades'         => $this->grades,
            'types'          => $this->types,
        ]);
    }

    /**
     * Update a template
     */
    public function update(Request $request, RemarkTemplate $remarkTemplate)
    {
        $validated = $request->validate([
            'phase_id'    => 'required|exists:phases,id',
            'grade'       => 'required|in:A,B,C,D,E,F',
            'type'        => 'required|in:PL_COMD,OC,CI,COMDT',
            'remark_text' => 'required|string',
        ]);

        $remarkTemplate->update($validated);

        return redirect()
            ->route('remark-templates.index', ['phase_id' => $validated['phase_id']])
            ->with('success', 'Remark template updated successfully.');
    }

    /**
     * Delete a single template
     */
    public function destroy(RemarkTemplate $remarkTemplate)
    {
        $phaseId = $remarkTemplate->phase_id;
        $remarkTemplate->delete();

        return redirect()
            ->route('remark-templates.index', ['phase_id' => $phaseId])
            ->with('success', 'Remark template deleted.');
    }

    /**
     * Bulk delete
     */
    public function bulkDelete(Request $request)
    {
        $request->validate([
            'ids'   => 'required|array',
            'ids.*' => 'exists:remark_templates,id',
        ]);

        RemarkTemplate::whereIn('id', $request->ids)->delete();

        return redirect()->route('remark-templates.index')
            ->with('success', 'Selected remark templates deleted.');
    }
}