<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Exception;

class SsoController extends Controller
{
    // ==================== CONFIGURATION ====================
    
    // MCOP URLs - Update these to match your production environment
    // private const MCOP_BASE_URL = 'http://173.208.155.231';
    private const MCOP_BASE_URL = 'http://127.0.0.1:8000';
    private const MCOP_LOGIN_URL = self::MCOP_BASE_URL . '/mcop';
    private const MCOP_LOGOUT_URL = self::MCOP_BASE_URL . '/logout';
    private const MCOP_API_VALIDATE_TOKEN = self::MCOP_BASE_URL . '/api/sso/user';
    
    // Route names
    private const ROUTE_HOME = '/';
    private const ROUTE_DASHBOARD = 'dashboard';
    
    // Session keys
    private const SESSION_PENDING_MCOP_USER = 'pending_mcop_user';
    private const SESSION_PENDING_SSO_TOKEN = 'pending_sso_token';
    private const SESSION_PENDING_REDIRECT_URL = 'pending_redirect_url';
    private const SESSION_PENDING_LOGIN_TIME = 'pending_login_time';
    private const SESSION_SSO_AUTHENTICATED = 'sso_authenticated';
    private const SESSION_SSO_TOKEN = 'sso_token';
    private const SESSION_MCOP_USER_DATA = 'mcop_user_data';
    private const SESSION_SSO_LOGIN_TIME = 'sso_login_time';
    private const SESSION_SSO_TARGET_APP = 'sso_target_app';
    private const SESSION_LOGIN_PROGRESS = 'login_progress';
    private const SESSION_LOGIN_STEP = 'login_step';
    
    // User type constants
    private const USER_TYPE_ADMIN = 'admin';
    private const USER_TYPE_DATA_ENTRY_CLERK = 'data_entry_clerk';
    
    // Admin rank patterns
    private const ADMIN_RANKS = ['COL', 'LT COL', 'MAJ', 'BRIG', 'GEN', 'LT GEN', 'MAJ GEN'];
    
    // Super user ranks (highest privileges)
    private const SUPER_USER_RANKS = ['GEN', 'LT GEN', 'BRIG'];
    
    // Default values
    private const DEFAULT_EMAIL_DOMAIN = '@army.mil';
    private const DEFAULT_TARGET_APP = 'cits';
    private const DEFAULT_USER_NAME = 'Unknown';
    private const DEFAULT_USER_TYPE = self::USER_TYPE_DATA_ENTRY_CLERK;
    private const DEFAULT_RANK = 'N/A';
    
    // Timeout settings
    private const API_TIMEOUT = 15; // seconds
    private const MAX_RETRY_ATTEMPTS = 2;

    // ==================== MAIN SSO HANDLER ====================
    
    /**
     * Handle SSO login/registration from MCOP
     * AUTOMATIC MODE: Auto-login with token, redirect to MCOP without token
     */
    public function handleSsoLogin(Request $request)
    {
        Log::info('🔐 SSO LOGIN HANDLER INITIATED', [
            'url' => $request->fullUrl(),
            'method' => $request->method(),
            'ip' => $request->ip(),
            'has_sso_token' => $request->has('sso_token') ? 'YES' : 'NO',
            'has_token_param' => $request->has('token') ? 'YES' : 'NO',
        ]);
        
        // STEP 1: Check if already logged in
        if (Auth::check()) {
            Log::info('✅ User already authenticated', [
                'user_id' => Auth::id(),
                'email' => Auth::user()->email
            ]);
            return redirect()->route(self::ROUTE_DASHBOARD);
        }
        
        // STEP 2: Get SSO token from request (try both 'sso_token' and 'token' params)
        $ssoToken = $request->get('sso_token') ?? $request->get('token');
        
        if (!$ssoToken) {
            Log::warning('❌ No SSO token found - redirecting to MCOP', [
                'all_params' => $request->all()
            ]);
            
            // Build MCOP URL with redirect back to current URL
            $currentUrl = urlencode($request->fullUrl());
            $mcopUrl = self::MCOP_LOGIN_URL . '?redirect=' . $currentUrl;
            
            Log::info('🔄 Redirecting to MCOP', ['mcop_url' => $mcopUrl]);
            
            return redirect($mcopUrl);
        }
        
        Log::info('🔑 SSO token detected', [
            'token_preview' => substr($ssoToken, 0, 20) . '...',
            'token_length' => strlen($ssoToken)
        ]);
        
        // STEP 3: Check if this is a loading page request (from the loading page auto-submit)
        if ($request->has('loading') && $request->loading == 'true') {
            Log::info('⏳ Processing from loading page');
            return $this->processSsoLogin($request);
        }
        
        // STEP 4: Validate token with MCOP API
        try {
            $validationResult = $this->validateTokenWithMcop($ssoToken);
            
            if (!$validationResult['success']) {
                Log::error('❌ Token validation failed', [
                    'error' => $validationResult['error']
                ]);
                
                return $this->redirectToMcopWithError(
                    $request,
                    $validationResult['error'] ?? 'Invalid or expired token. Please login again.'
                );
            }
            
            // STEP 5: Extract user data from validation response
            $mcopUser = $validationResult['user'];
            $tokenInfo = $validationResult['token_info'];
            
            Log::info('✅ Token validated successfully', [
                'army_number' => $mcopUser['army_number'] ?? 'N/A',
                'email' => $mcopUser['email'] ?? 'N/A',
                'full_name' => $mcopUser['full_name'] ?? 'N/A',
                'rank' => $mcopUser['rank'] ?? 'N/A',
                'department' => $mcopUser['department'] ?? 'N/A',
                'target_app' => $tokenInfo['target_app'] ?? 'N/A',
                'token_expires_at' => $tokenInfo['expires_at'] ?? 'N/A'
            ]);
            
            // STEP 6: Show loading page and store data in session
            return $this->showLoadingPage($request, $mcopUser, $ssoToken, $tokenInfo);
            
        } catch (Exception $e) {
            Log::error('💥 SSO processing exception', [
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return $this->redirectToMcopWithError(
                $request,
                'Authentication error. Please try again.'
            );
        }
    }

    // ==================== TOKEN VALIDATION ====================
    
    /**
     * Validate SSO token with MCOP API
     * Uses the new API endpoint format: /api/sso/user?token=xxx
     */
    private function validateTokenWithMcop(string $token, int $attempt = 1): array
    {
        Log::info('📡 Validating token with MCOP API', [
            'attempt' => $attempt,
            'max_attempts' => self::MAX_RETRY_ATTEMPTS,
            'api_endpoint' => self::MCOP_API_VALIDATE_TOKEN
        ]);
        
        try {
            // Make API request with token as query parameter
            $response = Http::timeout(self::API_TIMEOUT)
                ->withHeaders([
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ])
                ->get(self::MCOP_API_VALIDATE_TOKEN, [
                    'token' => $token
                ]);
            
            Log::info('📡 MCOP API response received', [
                'status_code' => $response->status(),
                'successful' => $response->successful(),
                'response_preview' => substr($response->body(), 0, 200)
            ]);
            
            // Check if request was successful
            if (!$response->successful()) {
                Log::error('📡 MCOP API request failed', [
                    'status' => $response->status(),
                    'body' => $response->body(),
                    'headers' => $response->headers()
                ]);
                
                // Retry logic
                if ($attempt < self::MAX_RETRY_ATTEMPTS) {
                    Log::info('🔄 Retrying API request', ['attempt' => $attempt + 1]);
                    sleep(1); // Wait 1 second before retry
                    return $this->validateTokenWithMcop($token, $attempt + 1);
                }
                
                return [
                    'success' => false,
                    'error' => 'Cannot connect to authentication server. Please try again.'
                ];
            }
            
            // Parse JSON response
            $data = $response->json();
            
            // Validate response structure
            if (!isset($data['success'])) {
                Log::error('📡 Invalid API response structure', [
                    'response' => $data
                ]);
                return [
                    'success' => false,
                    'error' => 'Invalid response from authentication server.'
                ];
            }
            
            // Check if API call was successful
            if (!$data['success']) {
                Log::error('❌ MCOP API returned failure', [
                    'message' => $data['message'] ?? 'Unknown error',
                    'response' => $data
                ]);
                
                return [
                    'success' => false,
                    'error' => $data['message'] ?? 'Token validation failed.'
                ];
            }
            
            // Extract user data
            if (!isset($data['data']['user'])) {
                Log::error('📡 Missing user data in response', [
                    'response' => $data
                ]);
                return [
                    'success' => false,
                    'error' => 'Invalid user data received.'
                ];
            }
            
            $user = $data['data']['user'];
            $tokenInfo = $data['data']['token_info'] ?? [];
            
            // Validate required user fields
            if (empty($user['army_number']) && empty($user['email'])) {
                Log::error('📡 Missing required user identifiers', [
                    'user_data' => $user
                ]);
                return [
                    'success' => false,
                    'error' => 'Invalid user data: missing army number and email.'
                ];
            }
            
            Log::info('✅ Token validation successful', [
                'user_id' => $user['id'] ?? 'N/A',
                'army_number' => $user['army_number'] ?? 'N/A'
            ]);
            
            return [
                'success' => true,
                'user' => $user,
                'token_info' => $tokenInfo
            ];
            
        } catch (Exception $e) {
            Log::error('💥 Token validation exception', [
                'error' => $e->getMessage(),
                'attempt' => $attempt
            ]);
            
            // Retry logic for exceptions
            if ($attempt < self::MAX_RETRY_ATTEMPTS) {
                Log::info('🔄 Retrying after exception', ['attempt' => $attempt + 1]);
                sleep(1);
                return $this->validateTokenWithMcop($token, $attempt + 1);
            }
            
            return [
                'success' => false,
                'error' => 'Connection error. Please try again.'
            ];
        }
    }

    // ==================== LOADING PAGE ====================
    
    /**
     * Show loading page before processing login
     */
    private function showLoadingPage(Request $request, array $mcopUser, string $ssoToken, array $tokenInfo)
    {
        Log::info('🔄 Showing loading page', [
            'user' => $mcopUser['army_number'] ?? $mcopUser['email'] ?? 'unknown'
        ]);
        
        // Store data in session for the processing step
        session([
            self::SESSION_PENDING_MCOP_USER => $mcopUser,
            self::SESSION_PENDING_SSO_TOKEN => $ssoToken,
            self::SESSION_PENDING_REDIRECT_URL => $request->fullUrl(),
            self::SESSION_PENDING_LOGIN_TIME => now(),
            'pending_token_info' => $tokenInfo
        ]);
        
        return view('sso.loading', [
            'mcopUser' => $mcopUser,
            'tokenInfo' => $tokenInfo
        ]);
    }
    
    // ==================== LOGIN PROCESSING ====================
    
    /**
     * Process the actual SSO login (called from loading page)
     */
    public function processSsoLogin(Request $request)
    {
        Log::info('⚙️ Processing SSO login');
        
        // Get pending data from session
        $mcopUser = session(self::SESSION_PENDING_MCOP_USER);
        $ssoToken = session(self::SESSION_PENDING_SSO_TOKEN);
        $tokenInfo = session('pending_token_info', []);
        
        if (!$mcopUser || !$ssoToken) {
            Log::error('❌ No pending login data found in session');
            return redirect(self::ROUTE_HOME)
                ->with('error', 'Session expired. Please try logging in again.');
        }
        
        try {
            // Use database transaction for data integrity
            DB::beginTransaction();
            
            // STEP 1: Find or create user
            $user = $this->findOrCreateUser($mcopUser);
            
            if (!$user) {
                DB::rollBack();
                Log::error('❌ Failed to create/find user');
                
                $this->clearPendingSession();
                return $this->redirectToMcopWithError(
                    $request,
                    'Account setup failed. Please contact administrator.'
                );
            }
            
            // STEP 2: Update user's SSO information
            $this->updateUserSsoData($user, $mcopUser, $ssoToken, $tokenInfo);
            
            DB::commit();
            
            // STEP 3: Auto-login the user
            Auth::login($user, true); // "true" enables "remember me"
            
            // STEP 4: Store SSO session data
            session([
                self::SESSION_SSO_AUTHENTICATED => true,
                self::SESSION_SSO_TOKEN => $ssoToken,
                self::SESSION_MCOP_USER_DATA => $mcopUser,
                self::SESSION_SSO_LOGIN_TIME => now(),
                self::SESSION_SSO_TARGET_APP => $tokenInfo['target_app'] ?? $mcopUser['department'] ?? self::DEFAULT_TARGET_APP
            ]);
            
            // STEP 5: Update last login timestamp
            $user->update([
                'last_sso_login' => now(),
                'last_mcop_login' => now()
            ]);
            
            Log::info('🎉 SSO LOGIN SUCCESSFUL', [
                'user_id' => $user->id,
                'email' => $user->email,
                'army_number' => $user->army_number,
                'name' => $user->name,
                'type' => $user->type,
                'rank' => $user->rank,
                'department' => $user->department
            ]);
            
            // Clear pending session data
            $this->clearPendingSession();
            
            // STEP 6: Redirect to dashboard with welcome message
            $welcomeName = $user->rank ? $user->rank . ' ' . $user->name : $user->name;
            
            return redirect()->route(self::ROUTE_DASHBOARD)
                ->with('success', 'Welcome back, ' . $welcomeName . '!');
            
        } catch (Exception $e) {
            DB::rollBack();
            
            Log::error('💥 SSO processing exception', [
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            $this->clearPendingSession();
            
            return $this->redirectToMcopWithError(
                $request,
                'Authentication error occurred. Please try again.'
            );
        }
    }
    
    // ==================== USER MANAGEMENT ====================
    
    /**
     * Find or create user automatically based on MCOP data
     */
    private function findOrCreateUser(array $mcopUser): ?User
    {
        $armyNumber = $mcopUser['army_number'] ?? null;
        $email = $mcopUser['email'] ?? null;
        
        if (!$armyNumber && !$email) {
            Log::error('🚫 No army number or email in MCOP data', [
                'mcop_data' => $mcopUser
            ]);
            return null;
        }
        
        Log::info('👤 Looking for user', [
            'army_number' => $armyNumber,
            'email' => $email
        ]);
        
        // Try to find existing user by army_number first, then email
        $user = null;
        
        if ($armyNumber) {
            $user = User::where('army_number', $armyNumber)->first();
            if ($user) {
                Log::info('✅ Found user by army number', ['user_id' => $user->id]);
            }
        }
        
        if (!$user && $email) {
            $user = User::where('email', $email)->first();
            if ($user) {
                Log::info('✅ Found user by email', ['user_id' => $user->id]);
            }
        }
        
        // If user exists, update and return
        if ($user) {
            $this->updateExistingUser($user, $mcopUser);
            return $user;
        }
        
        // Create new user
        return $this->createNewUser($mcopUser);
    }
    
    /**
     * Update existing user with MCOP data
     */
    private function updateExistingUser(User $user, array $mcopUser): void
    {
        Log::info('🔄 Updating existing user', ['user_id' => $user->id]);
        
        try {
            $updateData = [
                'sso_enabled' => true,
                'last_sso_login' => now(),
                'is_active' => $mcopUser['is_active'] ?? true,
            ];
            
            // Update fields if they exist in MCOP data
            $fieldsToUpdate = [
                'full_name', 'email', 'army_number', 'rank', 
                'unit', 'phone', 'department', 'profile_photo'
            ];
            
            foreach ($fieldsToUpdate as $field) {
                if (isset($mcopUser[$field]) && !empty($mcopUser[$field])) {
                    $updateData[$field] = $mcopUser[$field];
                }
            }
            
            // Update name from full_name if available
            if (isset($mcopUser['full_name'])) {
                $updateData['name'] = $mcopUser['full_name'];
            }
            
            // Update user type based on MCOP data
            $newUserType = $this->determineUserType($mcopUser);
            if ($newUserType !== $user->type) {
                Log::info('🔄 User type changed', [
                    'old_type' => $user->type,
                    'new_type' => $newUserType
                ]);
                $updateData['type'] = $newUserType;
            }
            
            $user->update($updateData);
            
            Log::info('✅ User updated successfully', [
                'user_id' => $user->id,
                'updated_fields' => array_keys($updateData)
            ]);
            
        } catch (Exception $e) {
            Log::warning('⚠️ Could not update all user fields', [
                'error' => $e->getMessage(),
                'user_id' => $user->id
            ]);
        }
    }
    
    /**
     * Create new user from MCOP data
     */
    private function createNewUser(array $mcopUser): ?User
    {
        Log::info('✨ Creating new user from MCOP data', [
            'army_number' => $mcopUser['army_number'] ?? 'N/A',
            'email' => $mcopUser['email'] ?? 'N/A'
        ]);
        
        try {
            $userType = $this->determineUserType($mcopUser);
            
            $userData = [
                'name' => $mcopUser['full_name'] ?? self::DEFAULT_USER_NAME,
                'full_name' => $mcopUser['full_name'] ?? self::DEFAULT_USER_NAME,
                'email' => $mcopUser['email'] ?? ($mcopUser['army_number'] . self::DEFAULT_EMAIL_DOMAIN),
                'army_number' => $mcopUser['army_number'] ?? null,
                'type' => $userType,
                'password' => Hash::make(Str::random(32)), // Random secure password
                'is_active' => $mcopUser['is_active'] ?? true,
                'sso_enabled' => true,
                'created_via_sso' => true,
                'sso_created_at' => now(),
                'last_sso_login' => now(),
                'mcop_authenticated' => true,
                'last_mcop_login' => now(),
            ];
            
            // Add optional fields
            $optionalFields = [
                'rank', 'unit', 'phone', 'department', 
                'profile_photo', 'address', 'date_of_birth'
            ];
            
            foreach ($optionalFields as $field) {
                if (isset($mcopUser[$field]) && !empty($mcopUser[$field])) {
                    $userData[$field] = $mcopUser[$field];
                }
            }
            
            $user = User::create($userData);
            
            Log::info('✅ New user created successfully', [
                'user_id' => $user->id,
                'army_number' => $user->army_number,
                'email' => $user->email,
                'type' => $userType,
                'rank' => $user->rank
            ]);
            
            return $user;
            
        } catch (Exception $e) {
            Log::error('❌ Failed to create user', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'mcop_data' => $mcopUser
            ]);
            return null;
        }
    }
    
    /**
     * Update user's SSO-specific data
     */
    private function updateUserSsoData(User $user, array $mcopUser, string $ssoToken, array $tokenInfo): void
    {
        try {
            $user->update([
                'last_sso_login' => now(),
                'sso_enabled' => true,
                'mcop_authenticated' => true,
                'last_mcop_login' => now(),
            ]);
            
            Log::info('✅ User SSO data updated', ['user_id' => $user->id]);
            
        } catch (Exception $e) {
            Log::warning('⚠️ Could not update SSO data', [
                'error' => $e->getMessage(),
                'user_id' => $user->id
            ]);
        }
    }
    
    // ==================== USER TYPE DETERMINATION ====================
    
    /**
     * Determine user type based on MCOP data
     * Priority: explicit type > role > rank > super user flags > default
     */
    private function determineUserType(array $mcopUser): string
    {
        // 1. Check explicit type field
        if (isset($mcopUser['type'])) {
            $type = strtolower($mcopUser['type']);
            if (in_array($type, [self::USER_TYPE_ADMIN, self::USER_TYPE_DATA_ENTRY_CLERK])) {
                Log::info('👤 User type from explicit type field', ['type' => $type]);
                return $type;
            }
        }
        
        // 2. Check super user flag
        if (isset($mcopUser['is_super_user']) && $mcopUser['is_super_user'] === true) {
            Log::info('👤 User type: admin (super user flag)');
            return self::USER_TYPE_ADMIN;
        }
        
        // 3. Check admin user flag
        if (isset($mcopUser['is_admin_user']) && $mcopUser['is_admin_user'] === true) {
            Log::info('👤 User type: admin (admin user flag)');
            return self::USER_TYPE_ADMIN;
        }
        
        // 4. Check by role
        if (isset($mcopUser['role'])) {
            $role = strtolower($mcopUser['role']);
            switch ($role) {
                case 'admin':
                case 'administrator':
                case 'system_admin':
                case 'super_admin':
                    Log::info('👤 User type: admin (from role)', ['role' => $role]);
                    return self::USER_TYPE_ADMIN;
                    
                case 'clerk':
                case 'data_entry':
                case 'data_entry_clerk':
                case 'user':
                    Log::info('👤 User type: data_entry_clerk (from role)', ['role' => $role]);
                    return self::USER_TYPE_DATA_ENTRY_CLERK;
            }
        }
        
        // 5. Check by rank (higher ranks = admin)
        if (isset($mcopUser['rank'])) {
            $rank = strtoupper(trim($mcopUser['rank']));
            
            // Check super user ranks
            foreach (self::SUPER_USER_RANKS as $superRank) {
                if (str_contains($rank, $superRank)) {
                    Log::info('👤 User type: admin (super user rank)', ['rank' => $rank]);
                    return self::USER_TYPE_ADMIN;
                }
            }
            
            // Check admin ranks
            foreach (self::ADMIN_RANKS as $adminRank) {
                if (str_contains($rank, $adminRank)) {
                    Log::info('👤 User type: admin (admin rank)', ['rank' => $rank]);
                    return self::USER_TYPE_ADMIN;
                }
            }
        }
        
        // 6. Check by email pattern
        if (isset($mcopUser['email'])) {
            $email = strtolower($mcopUser['email']);
            if (str_contains($email, 'admin') || str_contains($email, 'administrator')) {
                Log::info('👤 User type: admin (from email pattern)');
                return self::USER_TYPE_ADMIN;
            }
        }
        
        // 7. Default to data_entry_clerk
        Log::info('👤 User type: data_entry_clerk (default)');
        return self::DEFAULT_USER_TYPE;
    }
    
    // ==================== HELPER METHODS ====================
    
    /**
     * Clear pending session data
     */
    private function clearPendingSession(): void
    {
        session()->forget([
            self::SESSION_PENDING_MCOP_USER,
            self::SESSION_PENDING_SSO_TOKEN,
            self::SESSION_PENDING_REDIRECT_URL,
            self::SESSION_PENDING_LOGIN_TIME,
            'pending_token_info'
        ]);
        
        Log::info('🧹 Pending session data cleared');
    }
    
    /**
     * Redirect to MCOP with error message
     */
    private function redirectToMcopWithError(Request $request, string $error)
    {
        $currentUrl = urlencode($request->fullUrl());
        $mcopUrl = self::MCOP_LOGIN_URL . '?redirect=' . $currentUrl;
        
        Log::warning('🔄 Redirecting to MCOP with error', [
            'error' => $error,
            'mcop_url' => $mcopUrl
        ]);
        
        return redirect($mcopUrl)->with('error', $error);
    }
    
    // ==================== PUBLIC ENDPOINTS ====================
    
    /**
     * AJAX endpoint for loading progress (optional feature)
     */
    public function checkLoginProgress(Request $request)
    {
        $progress = session(self::SESSION_LOGIN_PROGRESS, 0);
        $step = session(self::SESSION_LOGIN_STEP, 'Starting...');
        
        return response()->json([
            'progress' => $progress,
            'step' => $step,
            'complete' => $progress >= 100
        ]);
    }
    
    /**
     * Check SSO status (for debugging/monitoring)
     */
    public function checkSsoStatus()
    {
        $authenticated = Auth::check();
        
        $response = [
            'sso_authenticated' => session(self::SESSION_SSO_AUTHENTICATED, false),
            'user_logged_in' => $authenticated,
            'timestamp' => now()->toIso8601String()
        ];
        
        if ($authenticated) {
            $user = Auth::user();
            $response['user_data'] = [
                'id' => $user->id,
                'army_number' => $user->army_number,
                'email' => $user->email,
                'name' => $user->name,
                'full_name' => $user->full_name,
                'type' => $user->type,
                'rank' => $user->rank,
                'department' => $user->department,
                'is_active' => $user->is_active,
                'sso_enabled' => $user->sso_enabled
            ];
        }
        
        return response()->json($response);
    }
    
    /**
     * Direct MCOP redirect (manual trigger)
     */
    public function redirectToMcop(Request $request)
    {
        $redirectUrl = urlencode($request->get('redirect_to', url()->current()));
        $mcopUrl = self::MCOP_LOGIN_URL . '?redirect=' . $redirectUrl;
        
        Log::info('🔄 Manual redirect to MCOP', ['mcop_url' => $mcopUrl]);
        
        return redirect($mcopUrl);
    }
    
    /**
     * SSO Logout - clear session and redirect to MCOP logout
     */
    public function ssoLogout(Request $request)
    {
        $userId = Auth::id();
        $userEmail = Auth::user()->email ?? 'N/A';
        
        Log::info('🚪 SSO Logout initiated', [
            'user_id' => $userId,
            'user_email' => $userEmail
        ]);
        
        // Clear all SSO session data
        session()->forget([
            self::SESSION_SSO_AUTHENTICATED,
            self::SESSION_SSO_TOKEN,
            self::SESSION_MCOP_USER_DATA,
            self::SESSION_SSO_LOGIN_TIME,
            self::SESSION_SSO_TARGET_APP,
            self::SESSION_PENDING_MCOP_USER,
            self::SESSION_PENDING_SSO_TOKEN,
            self::SESSION_PENDING_REDIRECT_URL,
            self::SESSION_PENDING_LOGIN_TIME,
            'pending_token_info'
        ]);
        
        // Logout user from Laravel
        Auth::logout();
        
        // Invalidate and regenerate session
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        
        Log::info('✅ SSO Logout complete', [
            'user_id' => $userId,
            'user_email' => $userEmail
        ]);
        
        // Redirect to MCOP logout page
        return redirect(self::MCOP_LOGOUT_URL)
            ->with('success', 'You have been logged out successfully.');
    }
    
    /**
     * Manual SSO login for testing (accepts token in request)
     */
    public function manualSsoLogin(Request $request)
    {
        $validated = $request->validate([
            'sso_token' => 'required|string|min:10'
        ]);
        
        Log::info('🔧 Manual SSO login attempt', [
            'token_preview' => substr($validated['sso_token'], 0, 20) . '...'
        ]);
        
        // Add token to request and process
        $request->merge(['token' => $validated['sso_token']]);
        
        return $this->handleSsoLogin($request);
    }
}