<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\DB;
use App\Models\Admission;
use App\Models\Assessment;
use App\Models\RemarkTemplate;
use App\Models\GpvReport;
use App\Models\GpvReportDetail;
use App\Models\Bn;
use App\Models\Coy;
use App\Models\Exercise;
use App\Models\ExamWeight;
use App\Models\Phase;
use App\Models\Pl;
use App\Models\PR;
use App\Models\CrAssessment;
use App\Models\Sec;
use Illuminate\Http\Request;
use Illuminate\Database\Eloquent\Collection;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Models\Remark;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\StudentsExport; 
use App\Models\School;
use App\Models\Course;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Illuminate\Support\Str;
use App\Models\Report;
use App\Models\Intakes;
use App\Services\PhaseConfigService;
class StudentPerformanceController extends Controller
{

   protected $phaseConfigService;

    public function __construct(PhaseConfigService $phaseConfigService)
    {
        $this->phaseConfigService = $phaseConfigService;
    }



        private const GRADE_SCALE = [
        'A' => ['min' => 90.0, 'comment' => 'EXCELLENT'],
        'B' => ['min' => 80.0, 'comment' => 'VERY GOOD'],
        'C' => ['min' => 70.0, 'comment' => 'GOOD'],
        'D' => ['min' => 60.0, 'comment' => 'SATISFACTORY'],
        'E' => ['min' => 50.0, 'comment' => 'FAIR'],
        'F' => ['min' => 0.0, 'comment' => 'POOR']
    ];
/**
 * Get all active exercises for a phase dynamically
 */
private function getPhaseExercises($phaseId)
{
    return Exercise::where('phase_id', $phaseId)
        ->orderBy('title')
        ->get()
        ->keyBy('title'); // Key by title for easy access
}

/**
 * Get all active subjects dynamically
 */
private function getActiveSubjects()
{
    return \App\Models\Subject::orderBy('code')
        ->get()
        ->keyBy('code'); // Key by code for easy access
}

/**
 * Process remark template with student data
 */
private function processRemarkTemplate($remarkTemplate, $student, $phase = null)
{
    if (!$remarkTemplate || !$remarkTemplate->remark_text) {
        return null;
    }

    // Get phase name if not passed
    if (!$phase) {
        $phase = Phase::where('is_active', 1)->first();
    }

    $text = $remarkTemplate->remark_text;

    $replacements = [
        // Student name
        '{{name}}'       => strtoupper($student->name),

        // Phase name
        '{{phase}}'      => $phase ? $phase->name : '',

        // Gender — lowercase
        '{{gender_he}}'  => $student->sex == 'M' ? 'he'  : 'she',
        '{{gender_his}}' => $student->sex == 'M' ? 'his' : 'her',
        '{{gender_him}}' => $student->sex == 'M' ? 'him' : 'her',

        // Gender — capital first letter
        '{{gender_He}}'  => $student->sex == 'M' ? 'He'  : 'She',
        '{{gender_His}}' => $student->sex == 'M' ? 'His' : 'Her',
        '{{gender_Him}}' => $student->sex == 'M' ? 'Him' : 'Her',
    ];

    return str_replace(array_keys($replacements), array_values($replacements), $text);
}

/**
 * Calculate dynamic practical scores based on database exercises
 */
public function calculatePracticalScoreDynamic($studentId, $phaseName = null) 
{
    $phase = $phaseName
        ? Phase::where('name', $phaseName)->first()
        : Phase::where('is_active', 1)->first();

    if (!$phase) {
        return 0;
    }

    // Get all exercises for this phase dynamically
    $exercises = Exercise::where('phase_id', $phase->id)->get();
    
    $totalPractical = 0;

    foreach ($exercises as $exercise) {
        // Get marks for this specific exercise
        $marks = Assessment::where('admission_id', $studentId)
            ->where('phase', $phase->name)
            ->where('type', 'Weekly')
            ->where('exercises', $exercise->title)
            ->pluck('marks');

        // Calculate average
        $average = $marks->count() > 0 ? $marks->sum() / $marks->count() : 0;
        
        // Apply weight
        $weightedScore = $average * $exercise->weight / 100;
        
        $totalPractical += $weightedScore;
    }

    return $totalPractical;
}

/**
 * Get dynamic exercise data for a student
 */
private function getDynamicExerciseData($studentId, $phase)
{
    $exercises = Exercise::where('phase_id', $phase->id)->get();
    $exerciseData = [];

    foreach ($exercises as $exercise) {
        $marks = Assessment::where('admission_id', $studentId)
            ->where('phase', $phase->name)
            ->where('exercises', $exercise->title)
            ->pluck('marks');

        $weeks = Assessment::where('admission_id', $studentId)
            ->where('phase', $phase->name)
            ->where('exercises', $exercise->title)
            ->pluck('week', 'code_name');

        $average = $marks->count() > 0 ? $marks->sum() / $marks->count() : 0;
        $weightedScore = $average * $exercise->weight / 100;

        $exerciseData[$exercise->title] = [
            'marks' => $marks->toArray(),
            'weeks' => $weeks,
            'average' => $average,
            'weight' => $exercise->weight,
            'total' => $weightedScore,
            'code' => $exercise->title
        ];
    }

    return $exerciseData;
}

/**
 * Get dynamic subject data for End of Phase exams
 */
private function getDynamicSubjectData($studentId, $phase)
{
    $subjects = \App\Models\Subject::all();
    $subjectData = [];

    foreach ($subjects as $subject) {
        $assessment = Assessment::where('admission_id', $studentId)
            ->where('phase', $phase->name)
            ->where('type', 'End of Phase Exam')
            ->where('subject_id', $subject->id)
            ->first();

        if ($assessment) {
            $subjectData[] = [
                'subject' => $subject,
                'marks' => $assessment->marks,
                'weighted_score' => ($assessment->marks * $subject->weight / 100)
            ];
        }
    }

    return $subjectData;
}

/**
 * Updated prepareReportData3 with dynamic subjects and exercises
 */
private function prepareReportData3Dynamic($armyNumber, $phase = 1)
{
    if (!$armyNumber) {
        return response()->json(['error' => 'Army number is required.'], 400);
    }
    
    $student = Admission::where('army_number', $armyNumber)->first();

    if (!$student) {
        return back()->with('error', 'Student not found');
    }
    
    $phase = Phase::where('is_active', 1)->first();
    
    if (!$phase) {
        return back()->with('error', 'No active phase found');
    }

    // Get all assessments
    $assessments = Assessment::where('admission_id', $student->id)
        ->where('phase', $phase->name)
        ->get();

    if ($assessments->isEmpty()) {
        return back()->with('error', 'No assessments found for this student');
    }

    // ============================================
    // DYNAMIC EXERCISES DATA
    // ============================================
    $exerciseData = $this->getDynamicExerciseData($student->id, $phase);
    
    // ============================================
    // DYNAMIC SUBJECTS DATA (End of Phase Exams)
    // ============================================
    $subjectData = $this->getDynamicSubjectData($student->id, $phase);

    // ============================================
    // WEEKLY TESTS CALCULATION
    // ============================================
    $weeklyMarks1 = Assessment::where('admission_id', $student->id)
        ->where('phase', $phase->name)
        ->where('assessment_group', 'Theory')
        ->where('type', 'Weekly')
        ->pluck('marks');

    $weeklyAverage = $weeklyMarks1->count() > 0 
        ? $weeklyMarks1->sum() / $weeklyMarks1->count() 
        : 0;

    $weeklyWeight = ExamWeight::where('phase_id', $phase->id)
        ->where('name', 'Weekly')
        ->value('weight') ?? 10;

    $weeklyWeightedScore = ($weeklyAverage * $weeklyWeight) / 100;

    // ============================================
    // MONTHLY TESTS CALCULATION
    // ============================================
    $monthlyMarks1 = Assessment::where('admission_id', $student->id)
        ->where('phase', $phase->name)
        ->where('type', 'Monthly')
        ->where('assessment_group', 'Theory')
        ->pluck('marks');

    $monthlyAverage = $monthlyMarks1->count() > 0 
        ? $monthlyMarks1->sum() / $monthlyMarks1->count() 
        : 0;

    $monthlyWeight = ExamWeight::where('phase_id', $phase->id)
        ->where('name', 'Monthly')
        ->value('weight') ?? 10;

    $monthlyWeightedScore = ($monthlyAverage * $monthlyWeight) / 100;

    // ============================================
    // END OF PHASE EXAMS (Dynamic - No Duplicates)
    // ============================================
    $marks = Assessment::where('admission_id', $student->id)
        ->where('phase', $phase->name)
        ->where('type', 'End of Phase Exam')
        ->where('assessment_group', 'Theory')
        ->with(['subject'])
        ->get()
        ->unique('subject_id')
        ->values();

    // Calculate totals for End of Phase
    $totalMarks = 0;
    $totalWeightedScore = 0;

    foreach ($marks as $mark) {
        if ($mark->subject) {
            $totalMarks += $mark->marks;
            $totalWeightedScore += ($mark->marks * $mark->subject->weight / 100);
        }
    }

    // ============================================
    // END OF PHASE AVERAGE
    // ============================================
    $endOfPhase = Assessment::where('admission_id', $student->id)
        ->where('phase', $phase->name)
        ->where('type', 'End of Phase Exam')
        ->pluck('marks');
    
    $endOfPhaseAverage = $endOfPhase->count() > 0 
        ? $endOfPhase->sum() / $endOfPhase->count() 
        : 0;
    
    $endOfPhaseWeight = ExamWeight::where('phase_id', $phase->id)
        ->where('name', 'End of Phase Exam')
        ->value('weight') ?? 0;
    
    $totalEndOfPhase = $endOfPhaseAverage * $endOfPhaseWeight / 100;

    // ============================================
    // DYNAMIC PRACTICAL CALCULATION
    // ============================================
    $totalPractical = $this->calculatePracticalScoreDynamic($student->id, $phase->name);

    // Get exercise weights dynamically
    $exerciseWeights = Exercise::where('phase_id', $phase->id)
        ->pluck('weight', 'title')
        ->toArray();

    // Get exam weights
    $examWeights = ExamWeight::where('phase_id', $phase->id)
        ->pluck('weight', 'name')
        ->toArray();

    // Calculate performance data
    $performanceData = $this->calculatePerformanceData(
        $assessments, 
        $exerciseWeights, 
        $examWeights, 
        $phase
    );

    // Get admission and remarks
    $admission = Admission::findOrFail($student->id);
    $remarks = Remark::where('admission_id', $student->id)->get();


    // Weekly and Monthly detailed marks
    $weeklyMarks = Assessment::where('admission_id', $student->id)
        ->where('phase', $phase->name)
        ->where('type', 'Weekly')
        ->where('assessment_group', 'Theory')
        ->orderBy('week', 'asc')
        ->get();

    $monthlyMarks = Assessment::where('admission_id', $student->id)
        ->where('phase', $phase->name)
        ->where('type', 'Monthly')
        ->orderBy('week', 'asc')
        ->get();

    return [
        // Student info
        'student' => $student,
        'admission' => $admission,
        'phase' => $phase,
        'remarks' => $remarks,
        
        // Dynamic data (NEW!)
        'exerciseData' => $exerciseData,
        'subjectData' => $subjectData,
        'exercises' => Exercise::where('phase_id', $phase->id)->get(),
        'subjects' => \App\Models\Subject::all(),
        
        // Marks data
        'marks' => $marks,
        'totalMarks' => $totalMarks,
        'totalWeightedScore' => $totalWeightedScore,
        
        // Weekly/Monthly
        'weeklyAverage' => $weeklyWeightedScore,
        'monthlyAverage' => $monthlyWeightedScore,
        'weeklyScore' => $weeklyAverage,
        'monthlyScore' => $monthlyAverage,
        'weeklyMark' => $weeklyMarks,
        'monthlyMarks' => $monthlyMarks,
        
        // End of Phase
        'endOfPhaseAverage' => $endOfPhaseAverage,
        'totalEndOfPhase' => $totalEndOfPhase,
        
        // Practical total (dynamic)
        'totalPractical' => $totalPractical,
        
        // Performance data
        'performanceData' => $performanceData,
        'exerciseWeights' => $exerciseWeights,
        'examWeights' => $examWeights,
        
        // Metadata
        'whichPhase' => $phase
    ];
}

/**
 * Helper method to get exercise statistics
 */
private function getExerciseStatistics($studentId, $exerciseTitle, $phase)
{
    $marks = Assessment::where('admission_id', $studentId)
        ->where('phase', $phase->name)
        ->where('exercises', $exerciseTitle)
        ->pluck('marks');

    $weeks = Assessment::where('admission_id', $studentId)
        ->where('phase', $phase->name)
        ->where('exercises', $exerciseTitle)
        ->pluck('week', 'code_name');

    $exercise = Exercise::where('phase_id', $phase->id)
        ->where('title', $exerciseTitle)
        ->first();

    $average = $marks->count() > 0 ? $marks->sum() / $marks->count() : 0;
    $weight = $exercise ? $exercise->weight : 0;
    $total = $average * $weight / 100;

    return [
        'marks' => $marks->toArray(),
        'weeks' => $weeks,
        'average' => round($average, 2),
        'weight' => $weight,
        'total' => round($total, 2),
        'count' => $marks->count()
    ];
}



    public function getMaxScores()
{
    // Using the service to get max scores
    $maxScores = $this->phaseConfigService->getMaxScores();
    
    // Default values if the service returns empty data
    $defaultMaxScores = [
        'theory' => 0,
        'practical' => 0,
        '22cr' => 0,
        'pr' => 0
    ];
    
    // Return the max scores or fall back to defaults if empty
    return !empty($maxScores) ? $maxScores : $defaultMaxScores;
}

    public function search2()
    {

        return view('reports.search2');
    }



// Add this method to StudentPerformanceController
public function calculateStudentOverallScore($student, $phaseName = null)
{
    $phase = $phaseName
        ? Phase::where('name', $phaseName)->first()
        : Phase::where('is_active', 1)->first();

    if (!$phase) {
        return null;
    }

    $maxScores = $this->getMaxScores();
    
    $theoryScore = $this->calculateTheoryScore($student, $maxScores['theory'], $phase->name);
    $practicalScore = $this->calculatePracticalScore($student->id, $phase->name);
    $cr22Score = $this->calculate22CRScore($student, $maxScores['22cr'], $phase->name);
    $prScore = $this->calculatePrScore($student, $maxScores['pr'], $phase->name);

    $totalScore = $theoryScore + $practicalScore + $cr22Score + $prScore;

    return [
        'total' => $totalScore,
        'theory' => $theoryScore,
        'practical' => $practicalScore,
        'cr22' => $cr22Score,
        'pr' => $prScore,
        'grade' => $this->calculateGrade($totalScore),
        'comment' => $this->getComment($totalScore)
    ];
}

    
public function gpv(Request $request)
{
    try {
        // Set up filters
        $intakeId = $request->input('intake_id');
        $schoolId = $request->input('school_id');
        $courseId = $request->input('course_id');
        $phaseInput = $request->input('phase') ?? "Phase 1";
        $bnReq = $request->input('bn');
        $coyReq = $request->input('coy');
        $plReq = $request->input('pl');
        $gradeReq = $request->input('grade');
        $armyNumber = $request->input('army_number');

        // Fetch students with filters
        $students = Admission::withTrashed()
            ->where(function($query) {
                $query->where('is_decommissioned', 0)
                      ->orWhereNull('is_decommissioned');
            });
        $intakes = Intakes::all();

        // Apply filters if set
        if ($schoolId) {
            $students->where('school_id', $schoolId);
        }
        
        if ($armyNumber) {
            $students->where('army_number', 'like', "%$armyNumber%");
        }

        if ($courseId) {
            $students->where('course_id', $courseId);
        }

        if ($bnReq) {
            $students->where('bn', $bnReq);
        }
        
        if ($coyReq) {
            $students->where('coy', $coyReq);
        }
        
        if ($plReq) {
            $students->where('pl', $plReq);
        }

        if($intakeId) {
            $students->where('intake_id', $intakeId);
        }
        
        // Get the filtered results with eager loading
        $perPage = $request->input('per_page', 25);
        $students = $students->with(['school', 'course'])->paginate($perPage);

        // Get schools and courses for filter dropdowns
        $schools = School::orderBy('name')->get();
        $courses = Course::orderBy('name')->get();
        $bns = Bn::get();
        $coys = Coy::get();
        $pls = Pl::get();

        // Prepare student data for table
        $studentsData = [];
        $maxScores = [];

        // Get the phase based on the input
        if ($phaseInput) {
            $phase = Phase::where('name', $phaseInput)->first();
        } else {
            $phase = Phase::where('is_active', 1)->first();
        }

        $maxScores = $this->getMaxScores();

        if ($students->isEmpty()) {
            return back()->with('error', 'No students found for the selected filters.');
        }

        foreach ($students as $student) {
            $theoryScore = $this->calculateTheoryScore($student, $maxScores['theory'], $phase->name);
            $practicalScore = $this->calculatePracticalScore($student->id, $phase->name);
            $cr22Score = $this->calculate22CRScore($student, $maxScores['22cr'], $phase->name);
            $prScore = $this->calculatePrScore($student, $maxScores['pr'], $phase->name);

            $totalScore = $theoryScore + $practicalScore + $cr22Score + $prScore;
            $grade = $this->calculateGrade($totalScore);

            // Apply grade filter if set
            if ($gradeReq && strtolower($gradeReq) !== strtolower($grade)) {
                continue;
            }

            $comment = $this->getComment($totalScore);
            $remarks = $totalScore >= 50 ? 'PASSED' : 'FAILED';

            $studentsData[] = [
                'student_id' => $student->id, // Add student ID for saving
                'admission_number' => $student->admission_id,
                'army_number' => $student->army_number,
                'rank' => $student->rank,
                'name' => $student->name,
                'school' => $student->school ? $student->school->name : 'N/A',
                'course' => $student->course ? $student->course->name : 'N/A',
                'theory' => number_format($theoryScore, 2),
                'practical' => number_format($practicalScore, 2),
                'cr_22' => number_format($cr22Score, 2),
                'pr' => number_format($prScore, 2),
                'total' => number_format($totalScore, 2),
                'grade' => $grade,
                'comment' => $comment,
                'remarks' => $remarks
            ];
        }

        // Handle Save Report Request
        if ($request->has('save_report')) {
            return $this->saveGpvReport($request, $studentsData, $phase);
        }

        // If the user wants to download the Excel
        if ($request->has('download_excel')) {
            // Get school and course names for the report
            $schoolName = $schoolId ? School::find($schoolId)->name : 'All Schools';
            $courseName = $courseId ? Course::find($courseId)->name : 'All Courses';
            $intakeName = $intakeId ? Intakes::find($intakeId)->year : 'All Intakes';
            
            // Create report metadata
            $reportMetadata = [
                'title' => 'General Performance Review (GPV) - ' . $phase->name,
                'school' => $schoolName,
                'course' => $courseName,
                'intake' => $intakeName,
                'phase' => $phase->name,
                'bn' => $bnReq ?? 'All',
                'coy' => $coyReq ?? 'All',
                'pl' => $plReq ?? 'All',
                'grade' => $gradeReq ?? 'All',
                'date_generated' => date('F d, Y H:i:s'),
                'total_students' => count($studentsData)
            ];
            
            // Generate filename
            $filename = 'gpv_report_';
            if ($schoolId) $filename .= strtolower(str_replace(' ', '_', $schoolName)) . '_';
            if ($courseId) $filename .= strtolower(str_replace(' ', '_', $courseName)) . '_';
            $filename .= date('Y-m-d_H-i-s') . '.xlsx';
            
            return Excel::download(
                new StudentsExport($studentsData, $reportMetadata), 
                $filename
            );
        }

        // If the user wants to download the PDF
        if ($request->has('download_pdf') || $request->has('preview_pdf')) {
            // Get school and course names for the report title
            $schoolName = $schoolId ? School::find($schoolId)->name : 'All Schools';
            $courseName = $courseId ? Course::find($courseId)->name : 'All Courses';
            
            // Create report title and metadata
            $reportTitle = 'General Performance Review (GPV) - ' . $phase->name;
            $reportFilters = [
                'School' => $schoolName,
                'Course' => $courseName,
                'Intake' => $intakeId ? Intakes::find($intakeId)->year : 'All Intakes',
                'Date Generated' => date('F d, Y')
            ];
            
            // Load the PDF view with the data
            $pdf = PDF::loadView('gpv.pdf', [
                'studentsData' => $studentsData,
                'phase' => $phase,
                'schoolName' => $schoolName,
                'courseName' => $courseName,
                'reportTitle' => $reportTitle,
                'reportFilters' => $reportFilters,
                'filters' => $request->all()
            ]);
            
            // Manually set public path for DomPDF if necessary
            $pdf->setOptions([
                'public_path' => realpath(base_path('public'))
            ]);
            
            // Check if preview is requested
            if ($request->has('preview_pdf')) {
                return $pdf->stream('gpv_report_preview.pdf');
            }
            
            // Generate filename
            $filename = 'gpv_report_';
            if ($schoolId) $filename .= strtolower(str_replace(' ', '_', $schoolName)) . '_';
            if ($courseId) $filename .= strtolower(str_replace(' ', '_', $courseName)) . '_';
            $filename .= date('Y-m-d_H-i-s') . '.pdf';
            
            return $pdf->download($filename);
        }

        // Otherwise, show the view with filter options
        return view('gpv.gpv', compact(
            'studentsData', 'phase', 'intakes', 'schools', 'courses', 
            'schoolId', 'courseId', 'bns', 'coys', 'pls', 
            'bnReq', 'coyReq', 'plReq', 'students', 'perPage', 'gradeReq'
        ));

    } catch (\Exception $e) {
        return back()->with('error', $e->getMessage());
    }
}

/**
 * Save GPV Report to Database
 */
private function saveGpvReport($request, $studentsData, $phase)
{
    try {
        DB::beginTransaction();

        // Generate report name
        $schoolName = $request->school_id ? School::find($request->school_id)->name : 'All Schools';
        $courseName = $request->course_id ? Course::find($request->course_id)->name : 'All Courses';
        $reportName = 'GPV_' . $phase->name . '_' . $schoolName . '_' . date('Y-m-d_H-i-s');

        // Create main report record
        $report = GpvReport::create([
            'report_name' => $reportName,
            'intake_id' => $request->intake_id,
            'school_id' => $request->school_id,
            'course_id' => $request->course_id,
            'phase' => $phase->name,
            'bn' => $request->bn,
            'coy' => $request->coy,
            'pl' => $request->pl,
            'grade' => $request->grade,
            'army_number' => $request->army_number,
            'total_students' => count($studentsData),
            'filters' => $request->all(),
            'created_by' => auth()->id()
        ]);

        // Save report details
        foreach ($studentsData as $student) {
            GpvReportDetail::create([
                'gpv_report_id' => $report->id,
                'admission_id' => $student['student_id'],
                'admission_number' => $student['admission_number'],
                'army_number' => $student['army_number'],
                'rank' => $student['rank'],
                'name' => $student['name'],
                'school' => $student['school'],
                'course' => $student['course'],
                'theory_score' => str_replace(',', '', $student['theory']),
                'practical_score' => str_replace(',', '', $student['practical']),
                'cr_22_score' => str_replace(',', '', $student['cr_22']),
                'pr_score' => str_replace(',', '', $student['pr']),
                'total_score' => str_replace(',', '', $student['total']),
                'grade' => $student['grade'],
                'comment' => $student['comment'],
                'remarks' => $student['remarks']
            ]);
        }

        DB::commit();

        return redirect()->route('gpv', $request->except('save_report'))
            ->with('success', "Report '{$reportName}' saved successfully with " . count($studentsData) . " students!");

    } catch (\Exception $e) {
        DB::rollBack();
        return back()->with('error', 'Failed to save report: ' . $e->getMessage());
    }
}

/**
 * View saved reports
 */
public function savedReports()
{
    $reports = GpvReport::with(['intake', 'school', 'course', 'creator'])
        ->orderBy('created_at', 'desc')
        ->paginate(20);

    return view('gpv.saved-reports', compact('reports'));
}

/**
 * View specific saved report
 */
public function viewSavedReport($id)
{
    $report = GpvReport::with(['details', 'intake', 'school', 'course', 'creator'])
        ->findOrFail($id);

    return view('gpv.view-report', compact('report'));
}

/**
 * Delete saved report
 */
public function deleteSavedReport($id)
{
    try {
        $report = GpvReport::findOrFail($id);
        $report->delete();

        return redirect()->route('gpv.saved-reports')
            ->with('success', 'Report deleted successfully!');
    } catch (\Exception $e) {
        return back()->with('error', 'Failed to delete report: ' . $e->getMessage());
    }
}
    


private function getComment($totalScore): string
{
    foreach (self::GRADE_SCALE as $criteria) {
        if ($totalScore >= $criteria['min']) {
            return $criteria['comment'];
        }
    }
    return self::GRADE_SCALE['F']['comment'];
}


    public function search3()
    {

        return view('reports.search3');
    }


    public function search()
    {






        return view('reports.search');
    }



    public static function getMarks($studentId, $group, $exercise)
    {
        $phase = Phase::where('is_active', 1)->first();

        return Assessment::where('admission_id', $studentId)
           ->where('phase',$phase->name)
            ->where('assessment_group', $group)
            ->where('exercises', $exercise)
            ->pluck('marks')
            ->toArray();
    }

  
    
    public function generate(Request $request)
    {
        try {
            $uniqueId = Str::uuid();
            $verificationUrl = route('report.verify', ['token' => $uniqueId]);
    
            // Save to database
            Report::create([
                'army_number' => $request->army_number,
                'token' => $uniqueId,
                'verified' => true,
                'issued_at' => now(),
                'expires_at' => now()->addYears(1),
            ]);
    
            // Generate base64 QR code
            $qrCodeImage = base64_encode(
                // QrCode::format('png')->size(200)->generate($verificationUrl)
                QrCode::size(300)->generate($verificationUrl)
            );

         
    
            $data = $this->prepareReportData($request->army_number, $phase = 1);
            $data['qr_code'] = $qrCodeImage;


            $pdf = PDF::loadView('reports.performance', $data)
                ->setOption('isHtml5ParserEnabled', true)
                ->setOption('isPhpEnabled', true);
    
            return $pdf->stream('performance_report.pdf');
    
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }
    
    
    

private function prepareReportData($armyNumber, $phase = 1)
{
    $phase = Phase::where('is_active', 1)->first();
    
    $student = Admission::where('army_number', $armyNumber)->first();

    if (!$student) {
        return back()->with('error', 'Student not found');
    }

    // Get all assessments for this student
    $assessments = Assessment::where('admission_id', $student->id)
        ->where('phase', $phase->name)
        ->get();

    if ($assessments == null) {
        return back()->with('error', 'No assessments found for this student');
    }

// ============================================
// DYNAMIC REMARK TEMPLATES — fetched by grade
// ============================================
$grade = $performanceData['summary']['grade'] ?? 'F';

$ciRemark     = RemarkTemplate::where('phase_id', $phase->id)->where('grade', $grade)->where('type', 'CI')->first();
$comdtRemark  = RemarkTemplate::where('phase_id', $phase->id)->where('grade', $grade)->where('type', 'COMDT')->first();
$plComdRemark = RemarkTemplate::where('phase_id', $phase->id)->where('grade', $grade)->where('type', 'PL_COMD')->first();
$ocRemark     = RemarkTemplate::where('phase_id', $phase->id)->where('grade', $grade)->where('type', 'OC')->first();

// Process placeholders — now passes $phase so {{phase}} is replaced
$ciRemarkText     = $this->processRemarkTemplate($ciRemark,     $student, $phase);
$comdtRemarkText  = $this->processRemarkTemplate($comdtRemark,  $student, $phase);
$plComdRemarkText = $this->processRemarkTemplate($plComdRemark, $student, $phase);
$ocRemarkText     = $this->processRemarkTemplate($ocRemark,     $student, $phase);

    // Get exercise weights
    $exerciseWeights = Exercise::where('phase_id', $phase->id)->pluck('weight', 'title')->toArray();

    // Get exam weights (Weekly, Monthly, End of Phase)
    $examWeights = ExamWeight::where('phase_id', $phase->id)->pluck('weight', 'name')->toArray();

    // Calculate performance metrics with weights
    $performanceData = $this->calculatePerformanceData($assessments, $exerciseWeights, $examWeights, 'Phase ' . $phase);

    // ============================================
    // WEEKLY TESTS - FIXED CALCULATION
    // ============================================
    $weeklyMarks = Assessment::where('admission_id', $student->id)
        ->where('phase', $phase->name)
        ->where('type', 'Weekly')
        ->where('assessment_group', 'Theory')
        ->get();

    $weeklyMarks1 = Assessment::where('admission_id', $student->id)
        ->where('phase', $phase->name)
        ->where('assessment_group', 'Theory')
        ->where('type', 'Weekly')
        ->pluck('marks');

    $weeklyAverage = $weeklyMarks1->count() > 0 ? $weeklyMarks1->sum() / $weeklyMarks1->count() : 0;
    $weeklyWeight = ExamWeight::where('phase_id', $phase->id)->where('name', 'Weekly')->pluck('weight')->first() ?? 0;
    
    // FIXED: This is the weighted score for display
    $weeklyWeightedScore = ($weeklyAverage * $weeklyWeight) / 100;

    // ============================================
    // MONTHLY TESTS - FIXED CALCULATION
    // ============================================
    $monthlyMarks = Assessment::where('admission_id', $student->id)
        ->where('phase', $phase->name)
        ->where('type', 'Monthly')
        ->get();

    $monthlyMarks1 = Assessment::where('admission_id', $student->id)
        ->where('phase', $phase->name)
        ->where('type', 'Monthly')
        ->pluck('marks');

    $monthlyAverage = $monthlyMarks1->count() > 0 ? $monthlyMarks1->sum() / $monthlyMarks1->count() : 0;
    $monthlyWeight = ExamWeight::where('phase_id', $phase->id)->where('name', 'Monthly')->pluck('weight')->first() ?? 0;
    
    // FIXED: This is the weighted score for display
    $monthlyWeightedScore = ($monthlyAverage * $monthlyWeight) / 100;

    // If no records are found, make sure it's an empty array
    $monthlyMarks = $monthlyMarks ?? [];

    // ============================================
    // PRACTICAL EXERCISES CALCULATIONS
    // ============================================
    $ptMarks = $this->getMarks($student->id, 'Practical', 'PT', $phase);
    $tacMarks = $this->getMarks($student->id, 'Practical', 'TAC', $phase);
    $saaMarks = $this->getMarks($student->id, 'Practical', 'SAA', $phase);
    $muMarks = $this->getMarks($student->id, 'Practical', 'MU', $phase);
    $drMarks = $this->getMarks($student->id, 'Practical', 'DR', $phase);

    // Calculate DR component
    $dr = Assessment::where('admission_id', $student->id)
        ->where('exercises', 'DR')
        ->where('phase', $phase->name)
        ->pluck('marks');

    $drAverage = $dr->count() > 0 ? $dr->sum() / $dr->count() : 0;
    $drWeight = Exercise::where('phase_id', $phase->id)->where('title', 'DR')->pluck('weight')->first() ?? 0;
    $totalDr = $drAverage * $drWeight / 100;

    $drWeek = Assessment::where('admission_id', $student->id)
        ->where('exercises', 'DR')
        ->where('phase', $phase->name)
        ->pluck('week');

    // ============================================
    // END OF PHASE EXAMS
    // ============================================
    $endOfPhaseMarks = Assessment::where('admission_id', $student->id)
        ->where('type', 'End of Phase Exam')
        ->where('phase', $phase->name)
        ->pluck('marks')
        ->toArray();

    // Calculate PT component
    $pt1 = Assessment::where('admission_id', $student->id)
        ->where('phase', $phase->name)
        ->where('exercises', 'PT')
        ->pluck('marks');

    $ptAverage = $pt1->count() > 0 ? $pt1->sum() / $pt1->count() : 0;
    $ptWeight = Exercise::where('phase_id', $phase->id)->where('title', 'PT')->pluck('weight')->first() ?? 0;
    $totalPt = $ptAverage * $ptWeight / 100;

    // Calculate SAA component
    $saa = Assessment::where('admission_id', $student->id)
        ->where('phase', $phase->name)
        ->where('exercises', 'SAA')
        ->pluck('marks');

    $saaAverage = $saa->count() > 0 ? $saa->sum() / $saa->count() : 0;
    $saaWeight = Exercise::where('phase_id', $phase->id)->where('title', 'SAA')->pluck('weight')->first() ?? 0;
    $totalSaa = $saaAverage * $saaWeight / 100;

    // Calculate TAC component
    $tac = Assessment::where('admission_id', $student->id)
        ->where('exercises', 'TAC')
        ->where('phase', $phase->name)
        ->pluck('marks');

    $tacAverage = $tac->count() > 0 ? $tac->sum() / $tac->count() : 0;
    $tacWeight = Exercise::where('phase_id', $phase->id)->where('title', 'TAC')->pluck('weight')->first() ?? 0;
    $totalTac = $tacAverage * $tacWeight / 100;

    // Calculate COIN/IS component
    $coinWeek = Assessment::where('admission_id', $student->id)
        ->where('exercises', 'COIN/IS')
        ->where('phase', $phase->name)
        ->pluck('week');

    $coin = Assessment::where('admission_id', $student->id)
        ->where('exercises', 'COIN/IS')
        ->where('phase', $phase->name)
        ->pluck('marks');

    $coinAverage = $coin->count() > 0 ? $coin->sum() / $coin->count() : 0;
    $coinWeight = Exercise::where('phase_id', $phase->id)->where('title', 'COIN/IS')->pluck('weight')->first() ?? 0;
    $totalCoin = $coinAverage * $coinWeight / 100;

    // Calculate MU component
    $mu = Assessment::where('admission_id', $student->id)
        ->where('exercises', 'MU')
        ->where('phase', $phase->name)
        ->pluck('marks');

    $muAverage = $mu->count() > 0 ? $mu->sum() / $mu->count() : 0;
    $muWeight = Exercise::where('phase_id', $phase->id)->where('title', 'MU')->pluck('weight')->first() ?? 0;
    $totalMu = $muAverage * $muWeight / 100;

    // Calculate End of Phase component
    $endOfPhase = Assessment::where('admission_id', $student->id)
        ->where('type', 'End of Phase Exam')
        ->where('phase', $phase->name)
        ->pluck('marks');

    $endOfPhaseAverage = $endOfPhase->count() > 0 ? $endOfPhase->sum() / $endOfPhase->count() : 0;
    $endOfPhaseWeight = ExamWeight::where('phase_id', $phase->id)->where('name', 'End of Phase Exam')->pluck('weight')->first() ?? 0;
    $totalEndOfPhase = $endOfPhaseAverage * $endOfPhaseWeight / 100;

    $endOfPhaseExams = Assessment::where('admission_id', $student->id)
        ->where('type', 'End of Phase Exam')
        ->where('phase', $phase->name)
        ->get();

    // ============================================
    // WEEK DATA FOR PRACTICAL EXERCISES
    // ============================================
    $ptWeek = Assessment::where('admission_id', $student->id)
        ->where('exercises', 'PT')
        ->where('phase', $phase->name)
        ->pluck('week', 'code_name');

    $saaWeek = Assessment::where('admission_id', $student->id)
        ->where('exercises', 'SAA')
        ->where('phase', $phase->name)
        ->pluck('week', 'code_name');

    $drWeek = Assessment::where('admission_id', $student->id)
        ->where('exercises', 'DR')
        ->where('phase', $phase->name)
        ->pluck('week', 'code_name');

    $tacWeek = Assessment::where('admission_id', $student->id)
        ->where('exercises', 'Tac')
        ->where('phase', $phase->name)
        ->pluck('week', 'code_name');

    $muWeek = Assessment::where('admission_id', $student->id)
        ->where('exercises', 'MU')
        ->where('phase', $phase->name)
        ->pluck('week', 'code_name');

    // ============================================
    // COMMAND STRUCTURE DATA
    // ============================================
    $bn_id = Bn::where('name', $student->bn)->pluck('id')->first(); 
    $coy = Coy::where('name', $student->coy)->first();
    $oc = $coy->comd_rank.'| '.$coy->comd_name;

    $pl = Pl::where('coy_id', $coy->id)->where('bn_id', $bn_id)->where('name', $student->pl)->first();
    $pl_commad = $pl->comd_rank.'| '.$pl->comd_name;
    $pl_sgt = $pl->pl_sgt_rank.'| '.$pl->pl_sgt;

    $sec = Sec::where('pl_id', $pl->id)->where('coy_id', $coy->id)->where('bn_id', $bn_id)->where('name', $student->sec)->first();
    $sec_commad = $sec ? ($sec->comd_rank.'| '.$sec->comd_name) : 'N/A';

// ============================================
// REMARKS AND PHASE DATA
// ============================================
$remarks = Remark::where('admission_id', $student->id)->get();
$whichPhase = Phase::where('is_active', 1)->first();

// ============================================
// DYNAMIC REMARK TEMPLATES
// ============================================
// Get the grade from performance data (you already calculate this)
$assessments = Assessment::where('admission_id', $student->id)
    ->where('phase', $phase->name)
    ->get();

$exerciseWeights = Exercise::where('phase_id', $phase->id)
    ->pluck('weight', 'title')
    ->toArray();

$examWeights = ExamWeight::where('phase_id', $phase->id)
    ->pluck('weight', 'name')
    ->toArray();

$performanceData = $this->calculatePerformanceData(
    $assessments, 
    $exerciseWeights, 
    $examWeights
);

$grade = $performanceData['summary']['grade'] ?? 'F';

// Fetch dynamic remarks from database
$ciRemark = RemarkTemplate::where('phase_id', $phase->id)
    ->where('grade', $grade)
    ->where('type', 'CI')
    ->first();

$comdtRemark = RemarkTemplate::where('phase_id', $phase->id)
    ->where('grade', $grade)
    ->where('type', 'COMDT')
    ->first();

$plComdRemark = RemarkTemplate::where('phase_id', $phase->id)
    ->where('grade', $grade)
    ->where('type', 'PL_COMD')
    ->first();

$ocRemark = RemarkTemplate::where('phase_id', $phase->id)
    ->where('grade', $grade)
    ->where('type', 'OC')
    ->first();

// Process the remark text with student-specific data
$ciRemarkText = $this->processRemarkTemplate($ciRemark, $student);
$comdtRemarkText = $this->processRemarkTemplate($comdtRemark, $student);
$plComdRemarkText = $this->processRemarkTemplate($plComdRemark, $student);
$ocRemarkText = $this->processRemarkTemplate($ocRemark, $student);



    // ============================================
    // TEMPORARY DEBUG LOGGING
    // ============================================
    \Log::info('Weekly Tests Debug', [
        'student_id' => $student->id,
        'weeklyMarksCount' => $weeklyMarks1->count(),
        'weeklyMarksValues' => $weeklyMarks1->toArray(),
        'weeklyAverage' => $weeklyAverage,
        'weeklyWeight' => $weeklyWeight,
        'weeklyWeightedScore' => $weeklyWeightedScore
    ]);

    \Log::info('Monthly Tests Debug', [
        'student_id' => $student->id,
        'monthlyMarksCount' => $monthlyMarks1->count(),
        'monthlyMarksValues' => $monthlyMarks1->toArray(),
        'monthlyAverage' => $monthlyAverage,
        'monthlyWeight' => $monthlyWeight,
        'monthlyWeightedScore' => $monthlyWeightedScore
    ]);

    // ============================================
    // RETURN ALL DATA
    // ============================================
return [
    'whichPhase' => $whichPhase,
    'remarks' => $remarks,
    'phase' => $phase,
    // Add these new lines for dynamic remarks
    'ciRemarkText' => $ciRemarkText,
    'comdtRemarkText' => $comdtRemarkText,
    'plComdRemarkText' => $plComdRemarkText,
    'ocRemarkText' => $ocRemarkText,
    'performanceData' => $performanceData,
    'examWeights' => $examWeights,
    'coinWeek' => $coinWeek,
    'drWeek' => $drWeek,
    'drAverage' => $drAverage,
    'totalDr' => $totalDr,
    'dr' => $dr,
    'saaWeek' => $saaWeek,
    'tacWeek' => $tacWeek,
    'muWeek' => $muWeek,
    'oc' => $oc,
    'pl_commad' => $pl_commad,
    'pl_sgt' => $pl_sgt,
    'sec_commad' => $sec_commad,
    'ptWeek' => $ptWeek,
    'endOfPhaseExams' => $endOfPhaseExams,
    'endOfPhaseAverage' => $endOfPhaseAverage,
    'totalEndOfPhase' => $totalEndOfPhase,
    'totalPt' => $totalPt,
    'ptAverage' => $ptAverage,
    'totalSaa' => $totalSaa,
    'totalCoin' => $totalCoin,
    'coinAverage' => $coinAverage,
    'coinWeight' => $coinWeight,
    'coin' => $coin,
    'saaAverage' => $saaAverage,
    'tacAverage' => $tacAverage,
    'totalTac' => $totalTac,
    'muAverage' => $muAverage,
    'totalMu' => $totalMu,
    'endOfPhaseMarks' => $endOfPhaseMarks,
    'student' => $student,
    'weeklyMark' => $weeklyMarks,
    
    // FIXED: Clear and consistent variable names
    'weeklyWeightedScore' => $weeklyWeightedScore,    // Weighted score for display
    'weeklyRawAverage' => $weeklyAverage,             // Raw average for reference
    'monthlyMarks' => $monthlyMarks,
    'monthlyWeightedScore' => $monthlyWeightedScore,  // Weighted score for display
    'monthlyRawAverage' => $monthlyAverage,           // Raw average for reference
    
    'ptMarks' => $ptMarks,
    'tacMarks' => $tacMarks,
    'saaMarks' => $saaMarks,
    'muMarks' => $muMarks,
    'drMarks' => $drMarks,
    'performanceData' => $performanceData
];
}


    
    public function generateMultiple(Request $request)
    {
        $armyNumbers = $request->army_numbers;
    
        if (!$armyNumbers || !is_array($armyNumbers) || empty($armyNumbers)) {
            return back()->with('error', 'Please provide valid army numbers');
        }
    
        $results = [];
        $errors = [];
        $phase = $request->phase ?? 1; // Get phase from request or default to 1
    
        foreach ($armyNumbers as $armyNumber) {
            try {
                // Use the same prepareReportData method as in single report generation
                $results[$armyNumber] = $this->prepareReportData($armyNumber, $phase);
            } catch (\Exception $e) {
                $errors[$armyNumber] = $e->getMessage();
            }
        }
    
        $pdf = PDF::loadView('reports.combined_reports', [
            'results' => $results,
            'errors' => $errors,
            'armyNumbers' => $armyNumbers,
            'reportType' => $request->report_type,
        ]);
    
        return $pdf->download('combined_reports.pdf');
    }
    
    
    

    public function generate2(Request $request)
    {
        try {
            $data = $this->prepareReportData($request->army_number, $phase = 1);
    
            $pdf = PDF::loadView('reports.performance2', $data);
    
            return $pdf->stream('performance_report.pdf');
            
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }


  

    private function prepareReportData3($armyNumber, $phase = 1)
    {
        // We'll use the provided $phase parameter instead of overriding it
        // Remove the line that was setting $phase = 1 and ignoring the parameter
    
    
        // If army_number is not provided, handle the error
        if (!$armyNumber) {
            // Handle the error appropriately
            return response()->json(['error' => 'Army number is required.'], 400);
        }
        
        $student = Admission::where('army_number', $armyNumber)->first();
    
        if (!$student) {
            return back()->with('error', 'Student not found');
        }
        
        $phase = Phase::where('is_active', 1)->first();
        
        // Get all assessments for this student
        $assessments = Assessment::where('admission_id', $student->id)
            ->where('phase', $phase->name)
            ->get();
    
        if ($assessments->isEmpty()) {
            return back()->with('error', 'No assessments found for this student');
        }
    
        // Get exercise weights
        $exerciseWeights = Exercise::where('phase_id',$phase->id)->pluck('weight', 'title')->toArray();
    
        // Get exam weights (Weekly, Monthly, End of Phase)
        $examWeights = ExamWeight::where('phase_id',$phase->id)->pluck('weight', 'name')->toArray();
    
        // Calculate performance metrics with weights - use the $phase parameter here
        $performanceData = $this->calculatePerformanceData($assessments, $exerciseWeights, $examWeights, $phase);
    
        $weeklyMarks = Assessment::where('admission_id', $student->id)
            ->where('phase',$phase->name)
            ->where('type', 'Weekly')
            ->where('assessment_group', 'Theory')
            ->orderBy('week', 'asc')
            ->pluck('marks')
            ->toArray(); // Ensure it's an array
    
        $weeklyMarks1 = Assessment::where('admission_id', $student->id)
              ->where('phase',$phase->name)
            ->where('assessment_group', 'Theory')
            ->where('type', 'Weekly')
            ->pluck('marks');
    
        $weeklyAverage = $weeklyMarks1->count() > 0 ? $weeklyMarks1->sum() / $weeklyMarks1->count() : 0;
    
        $weeklyWeight = ExamWeight::where('phase_id',$phase->id)->where('name', 'Weekly')->pluck('weight')->first() ?? 0;
    
        $totalWeekly = $weeklyAverage * $weeklyWeight / 100;
    
        $monthlyMarks = Assessment::where('phase',$phase->name)->where('admission_id', $student->id)
            ->where('phase',$phase->name)
            ->where('type', 'Monthly')
            ->orderBy('week', 'asc')
            ->pluck('marks')
            ->toArray(); // Ensure it's an array
    
        // If no records are found, make sure it's an empty array
        $monthlyMarks = $monthlyMarks ?? [];
    
        // Calculate monthly component
        $monthlyMarks1 = Assessment::where('admission_id', $student->id)
        ->where('phase',$phase->name)
            ->where('type', 'Monthly')
            ->pluck('marks');
    
        $monthlyAverage = $monthlyMarks1->count() > 0 ? $monthlyMarks1->sum() / $monthlyMarks1->count() : 0;
    
        $monthlyWeight = ExamWeight::where('phase_id',$phase->id)->where('name', 'Monthly')->pluck('weight')->first() ?? 0;
        $totalMonthly = $monthlyAverage * $monthlyWeight / 100;
    
        // Use $phase instead of $request->phase in these method calls
        $ptMarks = $this->getMarks($student->id, 'Practical', 'PT', $phase);
        $tacMarks = $this->getMarks($student->id, 'Practical', 'TAC', $phase);
        $saaMarks = $this->getMarks($student->id, 'Practical', 'SAA', $phase);
        $muMarks = $this->getMarks($student->id, 'Practical', 'MU', $phase);
        $drMarks = $this->getMarks($student->id, 'Practical', 'DR', $phase);
    
        $endOfPhaseMarks = Assessment::where('admission_id', $student->id)
            ->where('phase',$phase->name)
            ->where('type', 'End of Phase Exam')
            ->pluck('marks')
            ->toArray(); // Ensure it's an array
    
        // Calculate PT component
        $pt1 = Assessment::where('admission_id', $student->id)
        ->where('phase',$phase->name)
            ->where('exercises', 'PT')
            ->pluck('marks');
    
        $ptAverage = $pt1->count() > 0 ? $pt1->sum() / $pt1->count() : 0;
    
        $ptWeight = Exercise::where('phase_id',$phase->id)->where('title', 'PT')->pluck('weight')->first() ?? 0;
        $totalPt = $ptAverage * $ptWeight / 100;
    
        $saa = Assessment::where('admission_id', $student->id)
           ->where('phase',$phase->name)
            ->where('exercises', 'SAA')
            ->pluck('marks');
    
        $saaAverage = $saa->count() > 0 ? $saa->sum() / $saa->count() : 0;
    
        $saaWeight = Exercise::where('phase_id',$phase->id)->where('title', 'SAA')->pluck('weight')->first() ?? 0;
        $totalSaa = $saaAverage * $saaWeight / 100;
    
        $tac = Assessment::where('admission_id', $student->id)
            ->where('phase',$phase->name)
            ->where('exercises', 'TAC')
            ->pluck('marks');
    
        $tacAverage = $tac->count() > 0 ? $tac->sum() / $tac->count() : 0;
    
        $tacWeight = Exercise::where('phase_id',$phase->id)->where('title', 'TAC')->pluck('weight')->first() ?? 0;
        $totalTac = $tacAverage * $tacWeight / 100;
    
        $mu = Assessment::where('admission_id', $student->id)
            ->where('phase',$phase->name)
            ->where('exercises', 'MU')
            ->pluck('marks');
    
        $muAverage = $mu->count() > 0 ? $mu->sum() / $mu->count() : 0;
    
        $muWeight = Exercise::where('phase_id',$phase->id)->where('title', 'MU')->pluck('weight')->first() ?? 0;
        $totalMu = $muAverage * $muWeight / 100;
    
        $endOfPhase = Assessment::where('admission_id', $student->id)
        ->where('phase',$phase->name)
            ->where('type', 'End of Phase Exam')
            ->pluck('marks');
    
        $endOfPhaseAverage = $endOfPhase->count() > 0 ? $endOfPhase->sum() / $endOfPhase->count() : 0;
    
        $endOfPhaseWeight = ExamWeight::where('phase_id',$phase->id)->where('name', 'End of Phase Exam')->pluck('weight')->first() ?? 0;
        $totalEndOfPhase = $endOfPhaseAverage * $endOfPhaseWeight / 100;
    
        $endOfPhaseExams = Assessment::where('admission_id', $student->id)
            ->where('phase',$phase->name)
            ->where('type', 'End of Phase Exam')
            ->get();
    
        $ptWeek = Assessment::where('admission_id', $student->id)
        ->where('phase',$phase->name)
            ->where('exercises', 'PT')
            ->pluck('week');
    
        $saaWeek = Assessment::where('admission_id', $student->id)
        ->where('phase',$phase->name)
            ->where('exercises', 'SAA')
            ->pluck('week');
    
        $tacWeek = Assessment::where('admission_id', $student->id)
        ->where('phase',$phase->name)
            ->where('exercises', 'Tac')
            ->pluck('week');
    
        $muWeek = Assessment::where('admission_id', $student->id)
        ->where('phase',$phase->name)
            ->where('exercises', 'MU')
            ->pluck('week');
    
        // Get student admission details
        $admission = Admission::findOrFail($student->id);
    
        // Get all assessments for the student
        $assessments = Assessment::where('admission_id', $student->id)
        ->where('phase',$phase->name)
            ->where('type', 'End of Phase Exam')
            ->with(['subject'])
            ->get()
            ->groupBy(['assessment_group', 'type']);
    
        // Get weights from models
        $exerciseWeights = Exercise::where('phase_id',$phase->id)->pluck('weight', 'title')->toArray();
        $examWeights = ExamWeight::where('phase_id',$phase->id)->pluck('weight', 'name')->toArray();
        $marks = $assessments['Theory']['End of Phase Exam'];
    
        return [
            'marks' => $marks,
            'saaWeek' => $saaWeek,
            'tacWeek' => $tacWeek,
            'muWeek' => $muWeek,
            'admission' => $admission,
            'assessments' => $assessments,
            'exerciseWeights' => $exerciseWeights,
            'examWeights' => $examWeights,
    
            'ptWeek' => $ptWeek,
            'endOfPhaseExams' => $endOfPhaseExams,
            'endOfPhaseAverage' => $endOfPhaseAverage,
            'totalEndOfPhase' => $totalEndOfPhase,
            'totalPt' => $totalPt,
            'ptAverage' => $ptAverage,
            'totalSaa' => $totalSaa,
            'saaAverage' => $saaAverage,
            'tacAverage' => $tacAverage,
            'totalTac' => $totalTac,
            'muAverage' => $muAverage,
            'totalMu' => $totalMu,
    
            'endOfPhaseMarks' => $endOfPhaseMarks,
            'student' => $student,
            'weeklyMark' => $weeklyMarks,
            'weeklyAverage' => $totalWeekly,
            'weeklyScore' => $weeklyAverage,
            'monthlyMarks' => $monthlyMarks,
            'monthlyAverage' => $totalMonthly,
            'monthlyScore' => $monthlyAverage,
            'ptMarks' => $ptMarks,
            'tacMarks' => $tacMarks,
            'saaMarks' => $saaMarks,
            'muMarks' => $muMarks,
            'drMarks' => $drMarks,
    
            'performanceData' => $performanceData
        ];
    }

/**
 * Generate transcript report with dynamic subjects and exercises
 */
public function generate3(Request $request)
{
    try {
        // Use the new dynamic method
        $data = $this->prepareReportData3Dynamic($request->army_number, $phase = 1);
        
        // Load the view with dynamic data
        $pdf = PDF::loadView('reports.transcript', $data);
        
        return $pdf->stream('performance_report.pdf');
        
    } catch (\Exception $e) {
        return back()->with('error', $e->getMessage());
    }
}



    private function calculatePerformanceData($assessments, $exerciseWeights, $examWeights)
    {
     // Get the active phase config directly from the service
    $data = $this->phaseConfigService->getPhaseConfig();
    
    // If you still need the phase name for some reason
    $phase = Phase::where('is_active', 1)->first();
    if (!$phase) {
        throw new \InvalidArgumentException("No active phase found.");
    }



    
        // Group assessments by type
        $mskTheoryAssessments = $assessments->where('type','Weekly')->where('assessment_group', 'Theory');



        $characterPrAssessments = $assessments->where('assessment_type', 'Character')
                                            ->where('type', 'PR');

        // Process MSK Theory assessments with weights
        if ($mskTheoryAssessments->isNotEmpty()) {
            $totalWeightedScore = 0;
            $totalAppliedWeight = 0;

            // Group by assessment_group (Weekly, Monthly, End of Phase)
            $groupedAssessments = $mskTheoryAssessments->groupBy('assessment_group');

            foreach ($groupedAssessments as $group => $groupAssessments) {
                // Get exam weight for this group (Weekly, Monthly, End of Phase)
                $examWeight = $examWeights[$group] ?? 0; // Default to 10 if not found

                // Group by exercise within each assessment_group
                $exerciseGroups = $groupAssessments->groupBy('exercises');

                foreach ($exerciseGroups as $exercise => $exerciseAssessments) {
                    // Get exercise weight
                    $exerciseWeight = $exerciseWeights[$exercise] ?? 1; // Default to 1 if not found

                    // Calculate average score for this exercise
                    $exerciseAvg = $exerciseAssessments->avg('marks');

                    // Apply both weights (exercise weight * exam weight)
                    $combinedWeight = $exerciseWeight * $examWeight;
                    $totalWeightedScore += $exerciseAvg * $combinedWeight;
                    $totalAppliedWeight += $combinedWeight;
                }
            }

            // Calculate weighted average
            $weightedAverage = $totalAppliedWeight > 0 ? $totalWeightedScore / $totalAppliedWeight : 0;

            // Scale to max_score (40%)

        }

        // Process MSK Practical assessments with weights (similar approach)


        // Process Character 22CR assessments with weights

            $student = $assessments->first()->admission;
            $score = $this->calculate22CRScore($student, $data['character_22cr']['max_score']);
            $data['character_22cr']['obtained'] = $score;





            // Add debug information
      
 


$theoryResults = $this->calculateTheoryScore($student, $data['msk_theory']['max_score']);
$data['msk_theory']['obtained'] = $theoryResults;

$practicalResults = $this->calculatePracticalScore($student->id);

$data['msk_practical']['obtained'] = $practicalResults;




// calculatePracticalScore

        // Process Character PR assessments
        if ($characterPrAssessments->isNotEmpty()) {
            $student = $assessments->first()->admission;
            $data['character_pr']['obtained'] = $this->calculatePrScore($student, $data['character_pr']['max_score']);
        }

        // Get student from the first assessment
        $student = $assessments->first()->admission;

        // Calculate PR score regardless of PR assessments
        $data['character_pr']['obtained'] = $this->calculatePrScore($student, $data['character_pr']['max_score']);

// return dd( $this->calculatePrScore($student, $data['character_pr']['max_score']));

        // Add debug information for PR calculations
       

        // Calculate totals and determine grade
        $totalObtained = $data['msk_theory']['obtained'] +
                        $data['msk_practical']['obtained'] +
                        $data['character_22cr']['obtained'] +
                        $data['character_pr']['obtained'];

        $totalMaxScore = $data['msk_theory']['max_score'] +
                        $data['msk_practical']['max_score'] +
                        $data['character_22cr']['max_score'] +
                        $data['character_pr']['max_score'];

        $totalPassMark = $data['msk_theory']['pass_mark'] +
                         $data['msk_practical']['pass_mark'] +
                         $data['character_22cr']['pass_mark'] +
                         $data['character_pr']['pass_mark'];

        // Determine overall status
        $mskPassed = $data['msk_theory']['obtained'] >= $data['msk_theory']['pass_mark'] &&
                    $data['msk_practical']['obtained'] >= $data['msk_practical']['pass_mark'];

        $characterPassed = $data['character_22cr']['obtained'] >= $data['character_22cr']['pass_mark'] &&
                          $data['character_pr']['obtained'] >= $data['character_pr']['pass_mark'];

        $overallPassed = $mskPassed && $characterPassed;

    // return dd($overallPassed);

        // Determine grade
        $percentage = ($totalObtained / $totalMaxScore) * 100;


        // return dd(round($percentage));
 
        $grade = $this->calculateGrade($percentage);



        // Add summary data
        $data['summary'] = [
            'total_obtained' => round($totalObtained, 2),
            'total_max_score' => $totalMaxScore,
            'total_pass_mark' => $totalPassMark,
            'percentage' => round($percentage, 1),
            'grade' => $grade,
           'status' => $this->hasPassed($percentage, 1) ? 'PASSED' : 'FAILED',
            'msk_status' => $mskPassed ? 'PASSED' : 'FAILED',
            'character_status' => $characterPassed ? 'PASSED' : 'FAILED'
        ];
        $data['msk_theory']['remarks'] = $this->getTheoryRemarks($this->calculateTheoryScore($student, $data['msk_theory']['max_score']));
        $data['msk_practical']['remarks'] = $this->getPracticalRemarks( $this->calculatePracticalScore($student->id));
        $data['character_22cr']['remarks'] = $this->get22CRRemarks($this->calculate22CRScore($student, $data['character_22cr']['max_score']));
        $data['character_pr']['remarks'] = $this->getPrRemarks( $this->calculatePrScore($student, $data['character_pr']['max_score']));

        return $data;
    }


    private function hasPassed(float $percentage): bool
{
    return $percentage >= 50;
}


   
public function calculateTheoryScore($student, $maxScore, $phaseName = null)
{
    // Get the phase either from input or fallback to active one
    $phase = $phaseName
        ? Phase::where('name', $phaseName)->first()
        : Phase::where('is_active', 1)->first();

    if (!$phase) {
        return 0;
    }

    // ============================================
    // WEEKLY TESTS - Average THEN apply weight
    // ============================================
 // ============================================
    // WEEKLY TESTS - Proper calculation
    // ============================================
    $weeklyMarks = Assessment::where('admission_id', $student->id)
        ->where('assessment_group', 'Theory')
        ->where('type', 'Weekly')
        ->where('phase', $phase->name)
        ->pluck('marks');

    $weeklyAverage = $weeklyMarks->count() > 0
        ? $weeklyMarks->sum() / $weeklyMarks->count()
        : 0;

    $weeklyWeight = ExamWeight::where('name', 'Weekly')
        ->where('phase_id', $phase->id)
        ->value('weight') ?? 15;

    // This is the actual weighted score for theory calculation
    $totalWeekly = ($weeklyAverage * $weeklyWeight) / 100;

    // ============================================
    // MONTHLY TESTS - Average THEN apply weight
    // ============================================
 $monthlyMarks = Assessment::where('admission_id', $student->id)
        ->where('assessment_group', 'Theory')
        ->where('type', 'Monthly')
        ->where('phase', $phase->name)
        ->pluck('marks');

    $monthlyAverage = $monthlyMarks->count() > 0
        ? $monthlyMarks->sum() / $monthlyMarks->count()
        : 0;

    $monthlyWeight = ExamWeight::where('phase_id', $phase->id)
        ->where('name', 'Monthly')
        ->value('weight') ?? 10;

    // This is the actual weighted score for theory calculation
    $totalMonthly = ($monthlyAverage * $monthlyWeight) / 100;

    // ============================================
    // END OF PHASE EXAMS
    // ============================================
    $endOfPhaseExams = Assessment::where('admission_id', $student->id)
        ->where('type', 'End of Phase Exam')
        ->where('assessment_group', 'Theory')
        ->where('phase', $phase->name)
        ->with('subject')
        ->get()
        ->unique('subject_id'); // Ensure no duplicate subjects

    // Calculate the weighted sum based on subject weights
    $endOfPhaseWeightedSum = 0;
    $totalSubjectWeight = 0;
    
    foreach ($endOfPhaseExams as $exam) {
        if ($exam->subject) {
            $subjectWeight = $exam->subject->weight ?? 0;
            $endOfPhaseWeightedSum += ($exam->marks * $subjectWeight);
            $totalSubjectWeight += $subjectWeight;
        }
    }

    // Get the End of Phase exam weight (should be 25%)
    $endOfPhaseWeight = ExamWeight::where('phase_id', $phase->id)
        ->where('name', 'End of Phase Exam')
        ->value('weight') ?? 25;

    // Scale to the correct End of Phase weight (25%)
    // Formula: (Sum of weighted marks / Total subject weights) * (End of Phase Weight / 100)
    $totalEndOfPhase = $totalSubjectWeight > 0 
        ? ($endOfPhaseWeightedSum / $totalSubjectWeight) * ($endOfPhaseWeight / 100)
        : 0;

    // ============================================
    // FINAL TOTAL
    // ============================================
   $total = $totalWeekly + $totalMonthly + $totalEndOfPhase;

    return round($total, 2);
}
public function calculate22CRScore($student, $maxScore, $phaseName = null)
{
    // // Get the phase either from input or fallback to active one
    $phase = $phaseName
        ? Phase::where('name', $phaseName)->first()
        : Phase::where('is_active', 1)->first();

    if (!$phase) {
        return 0;
    }

    // // 🔸 Simplified logic: fetch the score directly from `c_r_assessment2s`
    // $record = \DB::table('c_r_assessment2s')
    //     ->where('army_number', $student->army_number)
    //     ->where('phase_id', $phase->id)
    //     ->first();

    // if (!$record) {
    //     return 0;
    // }

    // // Directly return the score (or scale to $maxScore if needed)
    // return (float) $record->score;


   
    // Original 22CR computation logic — commented out
   

    // Get 22CR assessments for this student
    $assessments = CRAssessment::where('army_number', $student->army_number)
        ->with(['assessmentDetails.twentyTwoCR'])
        ->where('phase', $phase->name)
        ->get();

    if ($assessments->isEmpty()) {
        return 0;
    }

    $totalCRES = 0;

    foreach ($assessments as $assessment) {
        foreach ($assessment->assessmentDetails as $detail) {
            $f = $detail->f1 + $detail->f2 + $detail->f3 + $detail->f4 + $detail->f5;
            $fx = ($detail->f1 * 1) + ($detail->f2 * 2) + ($detail->f3 * 3) + ($detail->f4 * 4) + ($detail->f5 * 5);

            if ($f > 0) {
                $individualCRES = ($fx / $f) * $detail->twentyTwoCR->weight;
                $totalCRES += $individualCRES;
            }
        }
    }

    // Scale total CRES to max score
    $totalPossibleCRES = 720; // Assuming 720 is the maximum CRES
    return round(($totalCRES / $totalPossibleCRES) * $maxScore, 2);
   
}


public function calculatePrScore($student, $maxScore, $phaseName = null)
{
    // Get the phase either from input or fallback to active one
    $phase = $phaseName
        ? Phase::where('name', $phaseName)->first()
        : Phase::where('is_active', 1)->first();

    if (!$phase) {
        return 0;
    }

    // Get PR ratings for this student
    $prRatings = PR::where('phase', $phase->name)
        ->where('army_number', $student->army_number)
        ->get();

    if ($prRatings->isEmpty()) {
        return 0;
    }

    // // 🔸 Simplified temporary logic
    // // Take the average PR score (you can change this if you only want the latest)
    // $averageScore = $prRatings->avg('score');


    // // You can add more conditions here if needed
    // // e.g. elseif ($averageScore == 89.9) return 8.99;

    // // 🔸 For all other scores, just scale it directly (optional)
    // // or you can just return 0 if you want only the special rule to apply
    // return$averageScore;

  
  
    // Original PR weight-based calculation (commented out for now)
   

    $totalScore = $prRatings->sum('score');
    $totalPeers = Admission::where('bn', $student->bn)
        ->where('coy', $student->coy)
        ->where('pl', $student->pl)
        ->where('id', '!=', $student->id)
        ->count();

    $meanRating = $totalPeers > 0 ? $totalScore / $totalPeers : 0;

    if ($meanRating <= 0) {
        return 0;
    }

    $wholeNumber = floor($meanRating);
    $decimalPart = $meanRating - $wholeNumber;

    $assignedWeights = [
        1 => 6,
        2 => 10,
        3 => 14,
        4 => 18,
        5 => 22,
        6 => 26,
        7 => 30
    ];

    if (!isset($assignedWeights[$wholeNumber])) {
        return 0;
    }

    $baseWeight = $assignedWeights[$wholeNumber];
    $additionalWeight = $decimalPart * 4;
    $totalWeight = $baseWeight + $additionalWeight;

    return round(($totalWeight / 30) * $maxScore, 1);

}

    

 
/**
 * Get remarks for theory marks
 * 
 * @param float $marks
 * @return string
 */
private function getTheoryRemarks($marks)
{
    // Assuming the service is available as $this->phaseConfigService in your controller
    $passingCriteria = $this->phaseConfigService->getPassingCriteria();
    $passingMark = $passingCriteria['theory'] ?? 20;
    return $marks >= $passingMark ? 'PASSED' : 'FAILED';
}

/**
 * Get remarks for practical marks
 * 
 * @param float $marks
 * @return string
 */
private function getPracticalRemarks($marks)
{
    $passingCriteria = $this->phaseConfigService->getPassingCriteria();
    $passingMark = $passingCriteria['practical'] ?? 15;
    return $marks >= $passingMark ? 'PASSED' : 'FAILED';
}

/**
 * Get remarks for 22CR marks
 * 
 * @param float $marks
 * @return string
 */
private function get22CRRemarks($marks)
{
    $passingCriteria = $this->phaseConfigService->getPassingCriteria();
    $passingMark = $passingCriteria['22cr'] ?? 10;
    return $marks >= $passingMark ? 'PASSED' : 'FAILED';
}

/**
 * Get remarks for PR marks
 * 
 * @param float $marks
 * @return string
 */
private function getPrRemarks($marks)
{
    $passingCriteria = $this->phaseConfigService->getPassingCriteria();
    $passingMark = $passingCriteria['pr'] ?? 5;
    return $marks >= $passingMark ? 'PASSED' : 'FAILED';
}

// -----------------------------------------------------
// Optional: More efficient implementation using a single method
// -----------------------------------------------------

/**
 * Get remarks for any component
 * 
 * @param string $component Component name ('theory', 'practical', '22cr', 'pr')
 * @param float $marks
 * @return string
 */
private function getRemarks($component, $marks)
{
    $passingCriteria = $this->phaseConfigService->getPassingCriteria();
    
    // Default passing marks if not found in criteria
    $defaults = [
        'theory' => 20,
        'practical' => 15,
        '22cr' => 10,
        'pr' => 5
    ];
    
    $passingMark = $passingCriteria[$component] ?? $defaults[$component] ?? 0;
    return $marks >= $passingMark ? 'PASSED' : 'FAILED';
}




public function calculateExerciseWeightedMarks($admissionId, array $exerciseTitles, $type = 'Weekly', $assessmentGroup = 'Practical')
{
    $result = [];
    
    $phase = Phase::where('is_active', 1)->first();

    foreach ($exerciseTitles as $exerciseTitle) {
        // Get marks for the specific exercise
        $assessment = Assessment::where('admission_id', $admissionId)
         ->where('phase',$phase->name)
            ->where('type', $type)
            ->where('assessment_group', $assessmentGroup)
            ->where('exercises', $exerciseTitle)
            ->first();

        if (!$assessment) {
            $result[$exerciseTitle] = 0; // No marks found for this exercise
            continue;
        }

        $marks = $assessment->marks;

        // Get exercise weight
        $exercise = Exercise::where('phase_id',$phase->id)->where('title', $exerciseTitle)->first();

        if (!$exercise) {
            $result[$exerciseTitle] = 0; // Exercise not found
            continue;
        }

        $exerciseWeight = $exercise->weight;



        // Calculate weighted marks
        $weightedMarks = $marks * $exerciseWeight / 100;

        // Store in results array
        $result[$exerciseTitle] = $weightedMarks;
    }

    return $result;
}
public function calculatePracticalScore($studentId, $phaseName = null) {
    // Exercise groups with their respective weights
    // PT calculation
    // Get the phase either from input or fallback to active one
    $phase = $phaseName
        ? Phase::where('name', $phaseName)->first()
        : Phase::where('is_active', 1)->first();

    if (!$phase) {
        return 0; // or handle as needed
    }

    $pt1 = Assessment::where('admission_id', $studentId)
    ->where('phase',$phase->name)
        ->where('type', 'Weekly')
        ->where('exercises', 'PT')
        ->pluck('marks');

    $ptAverage = $pt1->count() > 0 ? $pt1->sum() / $pt1->count() : 0;
    $ptWeight = Exercise::where('phase_id',$phase->id)->where('title', 'PT')->pluck('weight')->first() ?? 0;
    $totalPt = $ptAverage * $ptWeight / 100;


    
    // SAA calculation
    $saa = Assessment::where('admission_id', $studentId)
        ->where('type', 'Weekly')
        ->where('phase',$phase->name)
        ->where('exercises', 'SAA')
        ->pluck('marks');
    $saaAverage = $saa->count() > 0 ? $saa->sum() / $saa->count() : 0;
    $saaWeight = Exercise::where('phase_id',$phase->id)->where('title', 'SAA')->pluck('weight')->first() ?? 0;
    $totalSaa = $saaAverage * $saaWeight / 100;



    
    // TAC calculation
    $tac = Assessment::where('admission_id', $studentId)
    ->where('phase',$phase->name)
        ->where('type', 'Weekly')
        ->where('exercises', 'TAC')
        ->pluck('marks');
    $tacAverage = $tac->count() > 0 ? $tac->sum() / $tac->count() : 0; 
    $tacWeight = Exercise::where('phase_id',$phase->id)->where('title', 'TAC')->pluck('weight')->first() ?? 0;
    $totalTac = $tacAverage * $tacWeight / 100;

  
    
    // MU calculation
    $mu = Assessment::where('admission_id', $studentId)
        ->where('phase',$phase->name)
        ->where('type', 'Weekly')
        ->where('exercises', 'MU')
        ->pluck('marks');

    $muAverage = $mu->count() > 0 ? $mu->sum() / $mu->count() : 0;
    $muWeight = Exercise::where('phase_id',$phase->id)->where('title', 'MU')->pluck('weight')->first() ?? 0;
    $totalMu = $muAverage * $muWeight / 100;
    
    // DR (Drill) calculation
    $dr = Assessment::where('admission_id', $studentId)
    ->where('phase',$phase->name)
        ->where('type', 'Weekly')
        ->where('exercises', 'DR')
        ->pluck('marks');

    $drAverage = $dr->count() > 0 ? $dr->sum() / $dr->count() : 0;
    $drWeight = Exercise::where('phase_id',$phase->id)->where('title', 'DR')->pluck('weight')->first() ?? 0;
    $totalDr = $drAverage * $drWeight / 100;

    
    
    // Total practical score calculation
    $totalPractical = $totalPt + $totalSaa + $totalTac + $totalMu + $totalDr;
    
    // Final score - 30% of the practical score
    // $total = $totalPractical * 0.3;


    
    return $total = $totalPractical ;
}




    private function fetchExerciseScores(int $admissionId, array $exerciseTitles, string $period): array
{
    return Assessment::where('admission_id', $admissionId)
        ->whereIn('exercise_title', $exerciseTitles)
        ->where('assessment_type', $period) // 'Weekly' or 'Monthly'
        ->pluck('score', 'exercise_title')
        ->toArray();
}




private function calculateGrade(float $percentage): string 
{
    foreach (self::GRADE_SCALE as $grade => $criteria) {
        if ($percentage >= $criteria['min']) {
            return $grade;
        }
    }
    return 'F';
}

function viewRemarks(){

    return view('reports.templates');

 
}


/**
 * Generate Transcript Reports for all filtered students
 */
public function generateAllTranscripts(Request $request)
{
    try {
        // Get filters from request
        $schoolId = $request->input('school_id');
        $courseId = $request->input('course_id');
        $intakeId = $request->input('intake_id');
        $bnReq = $request->input('bn');
        $coyReq = $request->input('coy');
        $plReq = $request->input('pl');
        $search = $request->input('search');

        // Fetch students with the same filters as your list method
        $students = Admission::withTrashed()
            ->where(function($query) {
                $query->where('is_decommissioned', 0)
                      ->orWhereNull('is_decommissioned');
            })
            ->with(['school', 'course', 'intake']);

        // Apply filters
        if ($schoolId) {
            $students->where('school_id', $schoolId);
        }
        
        if ($courseId) {
            $students->where('course_id', $courseId);
        }

        if ($intakeId) {
            $students->where('intake_id', $intakeId);
        }

        if ($bnReq) {
            $students->where('bn', $bnReq);
        }
        
        if ($coyReq) {
            $students->where('coy', $coyReq);
        }
        
        if ($plReq) {
            $students->where('pl', $plReq);
        }

        if ($search) {
            $students->where(function($query) use ($search) {
                $query->where('army_number', 'like', "%$search%")
                      ->orWhere('name', 'like', "%$search%");
            });
        }

        $students = $students->get();

        if ($students->isEmpty()) {
            return back()->with('error', 'No students found for the selected filters.');
        }

        // Prepare data for all students
        $allStudentsData = [];
        $errors = [];

        foreach ($students as $student) {
            try {
                $studentData = $this->prepareReportData3Dynamic($student->army_number);
                $allStudentsData[$student->army_number] = $studentData;
            } catch (\Exception $e) {
                $errors[$student->army_number] = $e->getMessage();
            }
        }

        // Generate filename based on filters
        $filename = 'transcript_reports_';
        if ($schoolId) {
            $schoolName = School::find($schoolId)->name ?? 'school';
            $filename .= strtolower(str_replace(' ', '_', $schoolName)) . '_';
        }
        if ($courseId) {
            $courseName = Course::find($courseId)->name ?? 'course';
            $filename .= strtolower(str_replace(' ', '_', $courseName)) . '_';
        }
        $filename .= date('Y-m-d_H-i-s') . '.pdf';

        // Load the combined view
        $pdf = PDF::loadView('reports.combined_transcripts', [
            'allStudentsData' => $allStudentsData,
            'errors' => $errors,
            'filters' => $request->all(),
            'totalStudents' => count($allStudentsData),
            'failedStudents' => count($errors)
        ])->setOption('isHtml5ParserEnabled', true)
          ->setOption('isPhpEnabled', true);

        return $pdf->stream($filename);

    } catch (\Exception $e) {
        return back()->with('error', 'Error generating reports: ' . $e->getMessage());
    }
}
}
