<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;
use App\Models\User;

class UserController extends Controller
{


  // In UserController
public function index()
{
    // Fetch all users
    $users = User::all();
    
    // Pass users to the view
    return view('users.index', compact('users'));
}


     // Show the user creation form
     public function create()
     {
         return view('users.create');
     }
 
     // Store a new user
     public function store(Request $request)
     {
         $request->validate([
             'name' => 'required|string|max:255',
             'email_prefix' => 'required|string',
             'email' => 'required|email|unique:users,email',
             'type' => 'required|in:admin,data_entry_clerk',
             'password' => 'required|string|min:8|confirmed',
         ]);
     
         $user = User::create([
             'name' => $request->name,
             'email' => $request->email,
             'type' => $request->type,
             'password' => Hash::make($request->password),
         ]);
     
         return redirect()->route('users.index')
             ->with('success', 'User created successfully!');
     }
     public function edit($id)
     {
         $user = User::findOrFail($id);
         return view('users.edit', compact('user'));
     }
     
     public function update(Request $request, $id)
     {
         $user = User::findOrFail($id);
         
         $rules = [
             'name' => 'required|string|max:255',
             'type' => 'required|in:admin,data_entry_clerk',
         ];
     
         // Only validate password if it's provided
         if ($request->filled('password')) {
             $rules['password'] = 'required|string|min:8|confirmed';
         }
     
         $request->validate($rules);
     
         // Prepare update data
         $userData = [
             'name' => $request->name,
             'type' => $request->type,
         ];
     
         // Only update password if a new one is provided
         if ($request->filled('password')) {
             $userData['password'] = Hash::make($request->password);
         }
     
         $user->update($userData);
     
         return redirect()
             ->route('users.index')
             ->with('success', 'User updated successfully!');
     }
     // Delete a user
     public function destroy($id)
     {
         $user = User::findOrFail($id);
         $user->delete();
 
         return redirect()->route('users.index')->with('success', 'User deleted successfully!');
     }
    // Show the form to change password
    public function showPasswordChangeForm()
    {
        return view('password_change');  // Make sure to create the blade template for this view
    }

    // Handle the password change
    public function updatePassword(Request $request)
    {
        // Validate the input
        $request->validate([
            'current_password' => 'required|string',
            'new_password' => 'required|string|min:8|confirmed',
        ]);

        // Check if the current password matches
        if (!Hash::check($request->current_password, Auth::user()->password)) {
            throw ValidationException::withMessages([
                'current_password' => ['The current password is incorrect.'],
            ]);
        }

        // Update the user's password
        Auth::user()->update([
            'password' => Hash::make($request->new_password),
        ]);

        // Redirect back with success message
        return redirect()->route('dashboard')->with('success', 'Password updated successfully!');
    }
}
