<?php

namespace App\Imports;

use App\Models\Admission;
use App\Models\Student;
use App\Models\Subject;
use App\Models\School;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\SkipsOnError;
use Maatwebsite\Excel\Concerns\SkipsErrors;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\SkipsFailures;

class AdmissionsImport implements ToModel, WithHeadingRow, WithValidation, SkipsOnError, SkipsOnFailure
{
    use SkipsErrors, SkipsFailures;

    protected $importedCount = 0;
    protected $updatedCount = 0;
    protected $skippedCount = 0;

    // Add counter for new admissions
    protected $newAdmissionCounter = 0;
    protected $baseCount = 0;
    protected $year = null;
    protected $schoolCode = 'BMTS';

  public function __construct()
    {
        // Set the year once for all imports
        $this->year = now()->year;
        
        // Get the latest count from database for the year
        $latestAdmission = Admission::where('year_admission', $this->year)
            ->orderBy('id', 'desc')
            ->first();
            
        if ($latestAdmission && $latestAdmission->admission_id) {
            // Extract the number from the last part of admission_id
            $parts = explode('-', $latestAdmission->admission_id);
            $lastPart = end($parts);
            $this->baseCount = (int)$lastPart;
        } else {
            $this->baseCount = 0;
        }
    }

 /**
     * @param array $row
     * @return \Illuminate\Database\Eloquent\Model|null
     */
    public function model(array $row)
    {
        // Check if a record with the same army number exists
        $existingAdmission = Admission::where('army_number', $row['army_number'])
            ->where('course_id', $row['course_id'])
            ->first();

        // Find or create the student
        $student = Student::firstOrCreate(
            ['army_number' => $row['army_number']],
            [
                'name' => $row['name'] ?? 'Unknown',
                'rank' => "O/CDT",
            ]
        );

        if ($existingAdmission) {
            // Update existing logic (unchanged)
            $needsUpdate = $this->needsUpdate($existingAdmission, $row);

            if ($needsUpdate) {
                $existingAdmission->update([
                    // ... your update fields
                ]);
                $this->updatedCount++;
                return null;
            } else {
                $this->skippedCount++;
                return null;
            }
        }

        // Generate unique admission_id for new record
        $this->newAdmissionCounter++;
        $admissionId = $this->generateAdmissionId($row);

        // If no existing record, create a new one
        $this->importedCount++;
        
        return new Admission([
            'rank' => $row['rank'] ?? "O/CDT",
            'email' => $row['email'] ?? null,
            'phone_number' => $row['phone_number'] ?? null,
            'school_id' => $row['school_id'] ?? null,
            'course_id' => $row['course_id'] ?? null,
            'intake_id' => $row['intake_id'] ?? null,
            'name' => $row['name'] ?? 'Unknown',
            'army_number' => $row['army_number'],
            'bn' => $row['bn'] ?? null,
            'coy' => $row['coy'] ?? null,
            'pl' => $row['pl'] ?? null,
            'sec' => $row['sec'] ?? null,
            'home_district' => $row['home_district'] ?? null,
            'village' => $row['village'] ?? null,
            'next_kin' => $row['next_kin'] ?? null,
            'next_kin_phone' => $row['next_kin_phone'] ?? null,
            'next_kin_rel' => $row['next_kin_rel'] ?? null,
            'sex' => $row['sex'] ?? null,
            'nin' => $row['nin'] ?? null,
            'father_name' => $row['father_name'] ?? null,
            'father_phone' => $row['father_phone'] ?? null,
            'mother_name' => $row['mother_name'] ?? null,
            'mother_phone' => $row['mother_phone'] ?? null,
            'education_level' => $row['education_level'] ?? null,
            'spouse_name' => $row['spouse_name'] ?? null,
            'spouse_phone' => $row['spouse_phone'] ?? null,
            'shoe_size' => $row['shoe_size'] ?? null,
            'student_id' => $student->id,
            'year_admission' => $row['year_admission'] ?? $this->year,
            'admission_id' => $admissionId, // Set the pre-generated ID
        ]);
    }


    /**
     * Compare existing admission with new data to determine if update is needed
     */
    private function needsUpdate($existingAdmission, $newData): bool
    {
     $fieldsToCompare = [
            'rank', 'email', 'phone_number', 'name', 'bn', 'coy', 'pl', 'sec',
            'school_id', 'intake_id', 'home_district', 'village', 'next_kin',
            'next_kin_phone', 'next_kin_rel', 'sex', 'nin', 'father_name',
            'father_phone', 'mother_name', 'mother_phone', 'education_level',
            'spouse_name', 'spouse_phone', 'shoe_size'
        ];

        foreach ($fieldsToCompare as $field) {
            if (isset($newData[$field]) && 
                $newData[$field] !== null && 
                $newData[$field] !== $existingAdmission->$field) {
                return true;
            }
        }

        return false;
    }

 /**
     * Generate unique admission ID
     */
    protected function generateAdmissionId($row)
    {
        // Get school code
        if (isset($row['school_id'])) {
            $school = School::find($row['school_id']);
            $schoolCode = $school ? strtoupper(substr($school->name, 0, 4)) : 'BMTS';
        } else {
            $schoolCode = 'BMTS';
        }

        $year = substr($this->year, -2);
        $count = $this->baseCount + $this->newAdmissionCounter;
        $formattedCount = str_pad($count, 6, '0', STR_PAD_LEFT);

        return "{$schoolCode}-{$year}-MH-{$formattedCount}";
    }

    /**
     * Get import statistics
     */
    public function getImportStats(): array
    {
        return [
            'imported' => $this->importedCount,
            'updated' => $this->updatedCount,
            'skipped' => $this->skippedCount
        ];
    }

    /**
     * Validation rules
     */
public function rules(): array
{
    return [
        'army_number' => 'required',
        'course_id' => 'required|exists:courses,id',
        'school_id' => 'required|exists:schools,id',
        'intake_id' => 'required|exists:intakes,id',
        'name' => 'required',
        'rank' => 'required',
        'sex' => 'required|in:M,F',
    ];
}


}