<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
class CRAssessment extends Model
{
    protected $table = 'c_r_assessments';

    protected $fillable = [
        'army_number',
        'name',
        'school_id',
        'course_id',
        'intake_id',
        'phase',
        'assessment_type',
        'assessment_group',
        'total_cres',
        'percentage_score'
    ];

    // Add this relationship method
    public function assessmentDetails()
    {
        return $this->hasMany(CRAssessmentDetail::class, 'cr_assessment_id');
    }

    // Add this relationship method for twenty_two_c_r_s
    public function twentyTwoCR()
    {
        return $this->hasManyThrough(
            TwentyTwoCR::class,
            CRAssessmentDetail::class,
            'cr_assessment_id', // Foreign key on cr_assessment_details
            'id', // Foreign key on twenty_two_c_r_s
            'id', // Local key on c_r_assessments
            'twenty_two_c_r_s_id' // Local key on cr_assessment_details
        );
    }
    public static function validateCRs($assessments)
    {
        if (empty($assessments)) {
            return [
                'valid' => false,
                'message' => 'No assessments provided',
                'missing' => range(1, 22),
                'extra' => []
            ];
        }
    
        // Get required CRs from database to ensure they exist
        $requiredCRs = TwentyTwoCR::pluck('id')->toArray();
        
        try {
            $submittedCRs = collect($assessments)->pluck('cr_id')
                ->map(function($id) {
                    return (int) $id;
                })
                ->sort()
                ->values()
                ->all();
    
            // Validate each rating
            foreach ($assessments as $assessment) {
                if (!isset($assessment['rating']) || 
                    !is_numeric($assessment['rating']) || 
                    $assessment['rating'] < 1 || 
                    $assessment['rating'] > 5) {
                    return [
                        'valid' => false,
                        'message' => "Invalid rating value for CR ID {$assessment['cr_id']}. Must be between 1 and 5.",
                        'cr_id' => $assessment['cr_id'],
                        'rating' => $assessment['rating'] ?? null
                    ];
                }
            }
    
            $missing = array_diff($requiredCRs, $submittedCRs);
            $extra = array_diff($submittedCRs, $requiredCRs);
            
            return [
                'valid' => empty($missing) && empty($extra),
                'message' => empty($missing) ? null : 'Missing required CRs: ' . implode(', ', $missing),
                'missing' => $missing,
                'extra' => $extra,
                'submitted' => $submittedCRs
            ];
        } catch (\Exception $e) {
            return [
                'valid' => false,
                'message' => 'Invalid assessment data structure: ' . $e->getMessage(),
                'missing' => $requiredCRs,
                'extra' => []
            ];
        }
    }


/**
 * The "booted" method of the model.
 * Adds a global scope to only show records for the active phase.
 *
 * @return void
 */
protected static function booted()
{
    // static::addGlobalScope('activePhase', function (Builder $builder) {
    //     $phase = \App\Models\Phase::where('is_active', 1)->first();
    //     if ($phase) {
    //         // Get the model's table name correctly from the builder
    //         $tableName = $builder->getModel()->getTable();
            
    //         // Qualify the column with the table name to avoid ambiguity
    //         $builder->where("$tableName.phase", $phase->name);
    //     }
    // });
}
}