<?php

namespace App\Traits;

use App\Models\ActivityLog;
use Illuminate\Support\Facades\Auth;

trait LogsActivity
{
    /**
     * The attributes that should be ignored when logging changes.
     *
     * @var array
     */
    protected $ignoreChangesTo = ['updated_at', 'created_at'];

    /**
     * Initialize the trait.
     */
    protected static function bootLogsActivity()
    {
        // Log model creation
        static::created(function ($model) {
            $logData = [
                'attributes' => $model->getAttributes(),
                'event_type' => 'creation'
            ];
            self::logActivity($model, 'created', $logData);
        });

        // Log model updates
        static::updated(function ($model) {
            $changes = self::getModelChanges($model);
            
            // Only log if there are meaningful changes
            if (!empty($changes)) {
                $changes['event_type'] = 'update';
                self::logActivity($model, 'updated', $changes);
            }
        });

        // Log model deletion
        static::deleted(function ($model) {
            $logData = [
                'attributes' => $model->getAttributes(),
                'event_type' => 'deletion'
            ];
            self::logActivity($model, 'deleted', $logData);
        });
    }

    /**
     * Get all relevant model changes.
     *
     * @param mixed $model
     * @return array
     */
    protected static function getModelChanges($model)
    {
        $changes = [];
        $dirty = $model->getDirty();
        $ignoreList = $model->ignoreChangesTo ?? ['updated_at', 'created_at'];
        
        foreach ($dirty as $key => $value) {
            // Skip ignored attributes
            if (in_array($key, $ignoreList)) {
                continue;
            }
            
            $original = $model->getOriginal($key);
            $changes['changes'][$key] = [
                'old' => $original,
                'new' => $value,
                'field' => $key,
                'field_type' => gettype($value)
            ];
        }

        // Add the full attributes for context if needed
        if (!empty($changes)) {
            $changes['attributes'] = $model->getAttributes();
            $changes['modified_fields'] = array_keys($changes['changes']);
            $changes['total_changes'] = count($changes['changes']);
        }
        
        return $changes;
    }

    /**
     * Log an activity record for the model.
     *
     * @param mixed $model
     * @param string $action
     * @param array $changes
     * @return void
     */
    protected static function logActivity($model, $action, $changes = [])
    {
        $user = Auth::user();
        
        // Get human-readable model name
        $modelName = class_basename(get_class($model));
        
        ActivityLog::create([
            'user_id' => $user ? $user->id : null,
            'user_name' => $user ? $user->name : 'system',
            'action' => $action,
            'model_type' => get_class($model),
            'model_name' => $modelName,
            'model_id' => $model->id,
            'changes' => $changes,
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'created_at' => now()
        ]);
    }

    /**
     * Get descriptive action name for the activity log.
     *
     * @param string $action
     * @return string
     */
    protected function getDescriptiveActionName($action)
    {
        $modelName = class_basename(get_class($this));
        
        return ucfirst($action) . ' ' . $modelName;
    }
}

